package nilvalidation

import (
	"errors"
	"fmt"
	"reflect"
	"strings"
)

// validates if a value is the same as its zero type
// does not check bools since they default to false
func isEmptyValue(v reflect.Value) bool {
	switch v.Kind() {
	case reflect.Array, reflect.Map, reflect.Slice, reflect.String:
		return v.Len() == 0
	case reflect.Int, reflect.Int8, reflect.Int16, reflect.Int32, reflect.Int64:
		return v.Int() == 0
	case reflect.Uint, reflect.Uint8, reflect.Uint16, reflect.Uint32, reflect.Uint64, reflect.Uintptr:
		return v.Uint() == 0
	case reflect.Float32, reflect.Float64:
		return v.Float() == 0
	case reflect.Interface, reflect.Ptr:
		return v.IsNil()
	}

	return false
}

// Validate returns an error if attributes are missing or nil defined
// i struct to validate
// maxDepth max recursive depth to check
// packageToCheck validate fields where its package is a subpackage of checkPackage; "" to check all packages
// toWhitelist nested struct paths to ignore
func Validate(i interface{}, maxDepth int, packageToCheck string, toWhitelist ...string) error {
	if maxDepth <= 0 {
		return fmt.Errorf("%d is not a valid max search depth", maxDepth)
	}

	if i == nil {
		return errors.New("cannot validate nil struct")
	}
	val := reflect.ValueOf(i)
	ty := reflect.TypeOf(i)
	switch val.Kind() {
	case reflect.Array, reflect.Map, reflect.Slice, reflect.String, reflect.Chan, reflect.Ptr:
		ty = ty.Elem()
	}

	whitelist := map[string]bool{}

	for _, e := range toWhitelist {
		whitelist[e] = true
	}

	return validateHelper(val, maxDepth, ty.Name(), packageToCheck, whitelist)
}

func validateHelper(values reflect.Value, depthLeft int, depthPath string, packageToCheck string, whitelist map[string]bool) error {
	if depthLeft == 0 {
		return nil
	}

	if values.Kind() == reflect.Ptr {
		return validateHelper(values.Elem(), depthLeft, depthPath, packageToCheck, whitelist)
	}

	fields := values.Type()

	var toCheck []reflect.Value
	var toCheckNames []string
	var missing []string

	for i := 0; i < values.Type().NumField(); i++ {
		field := fields.Field(i)
		value := values.FieldByName(field.Name)

		if whitelist[depthPath+"."+field.Name] {
			continue
		}

		if isEmptyValue(value) {
			missing = append(missing, depthPath+"."+field.Name)
		} else {
			for value.Kind() == reflect.Ptr {
				value = value.Elem()
			}
			shouldCheckPackage := strings.HasPrefix(value.Type().PkgPath(), packageToCheck)
			if ((value.Kind() == reflect.Struct) || (value.Kind() == reflect.Ptr)) && shouldCheckPackage {
				toCheck = append(toCheck, value)
				toCheckNames = append(toCheckNames, field.Name)
			}
		}
	}

	if len(missing) > 0 {
		return fmt.Errorf("missing: %s", strings.Join(missing, ", "))
	}

	for n := 0; n < len(toCheck); n++ {
		err := validateHelper(toCheck[n], depthLeft-1, depthPath+"."+toCheckNames[n], packageToCheck, whitelist)
		if err != nil {
			missing = append(missing, err.Error())
		}
	}
	if len(missing) > 0 {
		return errors.New(strings.Join(missing, ","))
	}

	return nil
}
