package nilvalidation

import (
	"net/http"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
)

type testStruct struct {
	a string
	b *secondTestStruct
	c int
	d float32
}

type secondTestStruct struct {
	aDepth2 string
	bDepth2 *thirdTestStruct
}

type thirdTestStruct struct {
	aDepth3 string
	bDepth3 float32
}

func TestValidate(t *testing.T) {
	t.Run("success", func(t *testing.T) {
		ts := &testStruct{
			a: "text",
			b: &secondTestStruct{
				aDepth2: "a",
				bDepth2: &thirdTestStruct{
					aDepth3: "a",
					bDepth3: 1,
				},
			},
			c: 1,
			d: 1.1,
		}

		err := Validate(ts, 5, "")
		assert.NoError(t, err)
	})

	t.Run("invalid param", func(t *testing.T) {
		err := Validate(&struct{}{}, 0, "")
		assert.EqualError(t, err, "0 is not a valid max search depth")

		err = Validate(nil, 1, "")
		assert.EqualError(t, err, "cannot validate nil struct")
	})

	t.Run("single depth level", func(t *testing.T) {
		t.Run("all missing", func(t *testing.T) {
			ts := new(testStruct)

			err := Validate(ts, 5, "")
			assert.EqualError(t, err, "missing: testStruct.a, testStruct.b, testStruct.c, testStruct.d")
		})

		t.Run("not a pointer", func(t *testing.T) {
			ts := testStruct{}

			err := Validate(ts, 5, "")
			assert.EqualError(t, err, "missing: testStruct.a, testStruct.b, testStruct.c, testStruct.d")
		})

		t.Run("zero values", func(t *testing.T) {
			ts := &testStruct{
				a: "",
				b: &secondTestStruct{},
				c: 0,
				d: 0,
			}

			err := Validate(ts, 5, "")
			assert.EqualError(t, err, "missing: testStruct.a, testStruct.c, testStruct.d")
		})

	})

	t.Run("two depth levels", func(t *testing.T) {
		t.Run("fail", func(t *testing.T) {
			ts := testStruct{
				a: "a",
				b: &secondTestStruct{},
				c: 1,
				d: 1.1,
			}

			err := Validate(ts, 5, "")
			assert.EqualError(t, err, "missing: testStruct.b.aDepth2, testStruct.b.bDepth2")
		})
		t.Run("multiple depth levels", func(t *testing.T) {
			type noPointerTestStruct struct {
				a string
				b secondTestStruct
				c int
			}

			ts := noPointerTestStruct{
				a: "a",
				b: secondTestStruct{},
				c: 1,
			}

			err := Validate(ts, 5, "")
			assert.EqualError(t, err, "missing: noPointerTestStruct.b.aDepth2, noPointerTestStruct.b.bDepth2")
		})
		t.Run("max depth 1", func(t *testing.T) {
			ts := testStruct{
				a: "a",
				b: &secondTestStruct{},
				c: 1,
				d: 1.1,
			}

			err := Validate(ts, 1, "")
			assert.NoError(t, err)
		})
	})
	t.Run("only check whitelisted packages", func(t *testing.T) {
		t.Run("package not present", func(t *testing.T) {
			ts := testStruct{
				a: "a",
				b: &secondTestStruct{},
				c: 1,
				d: 1.1,
			}

			err := Validate(ts, 5, "DNE")
			assert.NoError(t, err)
		})
		t.Run("package is present", func(t *testing.T) {
			type externalPackageTestStruct struct {
				a string
				b http.Client
				c strings.Builder
			}

			ts := externalPackageTestStruct{
				a: "a",
				b: http.Client{},
				c: strings.Builder{},
			}

			err := Validate(ts, 5, "net/http")
			assert.EqualError(t, err, "missing: externalPackageTestStruct.b.Transport, externalPackageTestStruct.b.Jar, externalPackageTestStruct.b.Timeout")

			err = Validate(ts, 5, "strings")
			assert.EqualError(t, err, "missing: externalPackageTestStruct.c.addr, externalPackageTestStruct.c.buf")
		})
		t.Run("nested pointers", func(t *testing.T) {
			type externalPackageTestStruct struct {
				a string
				b *http.Client
				c *strings.Builder
			}

			ts := externalPackageTestStruct{
				a: "a",
				b: &http.Client{},
				c: &strings.Builder{},
			}

			err := Validate(ts, 5, "net/http")
			assert.EqualError(t, err, "missing: externalPackageTestStruct.b.Transport, externalPackageTestStruct.b.Jar, externalPackageTestStruct.b.Timeout")

			err = Validate(ts, 5, "strings")
			assert.EqualError(t, err, "missing: externalPackageTestStruct.c.addr, externalPackageTestStruct.c.buf")
		})
		t.Run("double nested pointers", func(t *testing.T) {
			type externalPackageTestStruct struct {
				a string
				b **http.Client
				c **strings.Builder
			}

			h := &http.Client{}
			sb := &strings.Builder{}

			ts := externalPackageTestStruct{
				a: "a",
				b: &h,
				c: &sb,
			}

			err := Validate(ts, 5, "net/http")
			assert.EqualError(t, err, "missing: externalPackageTestStruct.b.Transport, externalPackageTestStruct.b.Jar, externalPackageTestStruct.b.Timeout")

			err = Validate(ts, 5, "strings")
			assert.EqualError(t, err, "missing: externalPackageTestStruct.c.addr, externalPackageTestStruct.c.buf")
		})
	})
	t.Run("thee layers of depth sanity test", func(t *testing.T) {
		t.Run("success", func(t *testing.T) {
			ts := &testStruct{
				a: "1",
				b: &secondTestStruct{
					aDepth2: "1",
					bDepth2: &thirdTestStruct{
						aDepth3: "1",
						bDepth3: 1,
					},
				},
				c: 1,
				d: 1,
			}
			err := Validate(ts, 5, "")
			assert.NoError(t, err)
		})
		t.Run("failure", func(t *testing.T) {
			ts := &testStruct{
				a: "1",
				b: &secondTestStruct{
					aDepth2: "1",
					bDepth2: &thirdTestStruct{},
				},
				c: 1,
				d: 1,
			}
			err := Validate(ts, 5, "")
			assert.EqualError(t, err, "missing: testStruct.b.bDepth2.aDepth3, testStruct.b.bDepth2.bDepth3")
		})
	})
	t.Run("whitelist", func(t *testing.T) {
		t.Run("single depth", func(t *testing.T) {
			ts := testStruct{
				a: "a",
				c: 1,
			}

			err := Validate(ts, 20, "")
			assert.EqualError(t, err, "missing: testStruct.b, testStruct.d")

			err = Validate(ts, 20, "", "testStruct.b")
			assert.EqualError(t, err, "missing: testStruct.d")

			err = Validate(ts, 20, "", "testStruct.b", "testStruct.d")
			assert.NoError(t, err)
		})
		t.Run("two depth levels", func(t *testing.T) {
			ts := testStruct{
				a: "a",
				b: &secondTestStruct{
					aDepth2: "1",
				},
				c: 1,
				d: 1,
			}

			err := Validate(ts, 20, "")
			assert.EqualError(t, err, "missing: testStruct.b.bDepth2")

			err = Validate(ts, 20, "", "testStruct.b.bDepth2")
			assert.NoError(t, err)
		})
	})
}
