package s2s2

import (
	"testing"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2/internal/authorization"
	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"code.justin.tv/amzn/TwitchS2S2/s2s2/mocks"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

func TestAuthorizatedSubject(t *testing.T) {
	cfg := &c7s.Config{
		IdentityOrigin:          "https://prod.s2s2identities.twitch.a2z.com",
		ServiceDomain:           "twitch",
		ServiceByIDAuthorityURI: "https://prod.services.s2s.twitch.a2z.com",
	}

	t.Run("Scope", func(t *testing.T) {
		as := &authorizedSubject{
			scope:   token.NewScope("abc"),
			tokenID: "123",
		}
		assert.Equal(t, as.scope, as.Scope())
		assert.Equal(t, as.tokenID, as.TokenID())
	})

	t.Run("authorizedSubject", func(t *testing.T) {
		t.Run("In", func(t *testing.T) {
			as := &authorizedSubject{
				Subject: authorization.NewSubject("https://prod.services.s2s.twitch.a2z.com/s2s-caller-service-id"),
			}
			t.Run("authorized", func(t *testing.T) {
				assert.True(t, as.In(service{ID: "s2s-caller-service-id", Cfg: cfg}))

				assert.True(t, as.In(
					service{
						ID:  "s2s-caller-service-id",
						Cfg: cfg,
					},
					diService{
						Name:  "s2s2-di-service",
						Stage: "prod",
						Cfg:   cfg,
					},
				))
			})

			t.Run("unauthorized", func(t *testing.T) {
				assert.False(t, as.In(diService{Name: "s2s-v0-id", Stage: "test", Cfg: cfg}))
				assert.False(t, as.In(service{ID: "s2s-v0-id2", Cfg: cfg}))
			})
		})
	})
	t.Run("distributed Identities", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		authSubject := mocks.NewMockAuthenticatedSubject(ctrl)
		authSubject.EXPECT().Service().Return("test-service").MinTimes(1)
		authSubject.EXPECT().Stage().Return("test").MinTimes(1)
		authSubject.EXPECT().Domain().Return("twitch").MinTimes(1)

		as := distributedIdentitiesAuthorizedSubject{
			cfg:                  cfg,
			AuthenticatedSubject: authSubject,
		}

		t.Run("ID", func(t *testing.T) {
			assert.Equal(t, "https://prod.s2s2identities.twitch.a2z.com/twitch/test-service/test.json", as.ID())
		})
		t.Run("In", func(t *testing.T) {
			t.Run("authorized", func(t *testing.T) {
				assert.True(t, as.In(&diService{Name: "test-service", Stage: "test", Cfg: cfg}))
				assert.True(t, as.In(
					diService{Name: "test-service", Stage: "test", Cfg: cfg},
					service{ID: "somethingelse", Cfg: cfg},
				))
				assert.True(t, as.In(
					diService{Name: "test-service", Stage: "test", Cfg: cfg},
					diService{Name: "another-test-service", Stage: "test", Cfg: cfg},
					service{ID: "somethingelse", Cfg: cfg},
				))

			})

			t.Run("unauthorized", func(t *testing.T) {
				assert.False(t, as.In(diService{Name: "test-service-2", Stage: "test", Cfg: cfg}))
				assert.False(t, as.In(diService{Name: "test-service", Stage: "prod", Cfg: cfg}))
				assert.False(t, as.In(service{ID: "test-service", Cfg: cfg}))
			})
		})
	})
}
