package s2s2_test

import (
	"fmt"
	"log"
	"net/http"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2/s2s2"
)

type server struct {
	AuthenticatedClient s2s2.HTTPClient
}

func (s *server) CreateBook(http.ResponseWriter, *http.Request) {}
func (s *server) GetBook(http.ResponseWriter, *http.Request)    {}
func (s *server) ListBooks(http.ResponseWriter, *http.Request)  {}

// Creating and using s2s clients
func ExampleS2S2() {
	s2s2, err := s2s2.New(&s2s2.Options{
		Config: &c7s.Config{
			// This is available via the s2s dashboard.
			ClientServiceName: "MyServiceName",
			// A comma separated list of origins this server is authoritative for. If your
			// service is served at https://prod.myserver.twitch.a2z.com, this should be
			// "https://prod.myserver.twitch.a2z.com"
			ServiceOrigins: "https://prod.myserver.twitch.a2z.com",
			// A list of s2s2 distributed identitied caller services and stage name.
			//
			// If you are unsure of what this is, see:
			// https://wiki.twitch.com/x/AKxkDQ
			AuthorizedDistributedIdentitiesServices: s2s2.DistributedIdentities(
				s2s2.DistributedIdentity{
					Service: "graphql",
					Stage:   "prod",
				},
			),
		},
	})
	if err != nil {
		panic(err)
	}

	server := &server{
		// AuthenticatedClient sends requests that are authenticated
		// the provided HTTP client is used to send authorized requests to the
		// target service.
		AuthenticatedClient: s2s2.HTTPClient(&http.Client{}),
	}

	mux := http.NewServeMux()
	// Calling createBook will now require the S2S capability createBook
	mux.Handle("createBook", s2s2.RequireScopes(http.HandlerFunc(server.CreateBook), s2s2.CapabilityScope("createBook")))
	// Calling getBook will now require the S2S capability getBook
	mux.Handle("getBook", s2s2.RequireScopes(http.HandlerFunc(server.GetBook), s2s2.CapabilityScope("getBook")))

	log.Fatal(http.ListenAndServe(":8080", mux))
}

func handleErr(error) {}

// RecordMetricsOnly mode will authenticate s2s2 callers that present
// authentication. If they don't, the request will still be processed,
// and s2s2 metrics will be logged to understand the service usage.
//
// Use OptionalRequestSubject to see if the RequestSubject is set to validate
// the authenticated caller.
func ExampleHandler_RecordMetricsOnly() {
	var s2s2Client *s2s2.S2S2

	myHandler := func(w http.ResponseWriter, r *http.Request) {
		if requestSubject, ok := s2s2.OptionalRequestSubject(r.Context()); ok {
			if _, err := w.Write([]byte(fmt.Sprintf("Hello %s!", requestSubject.ID()))); err != nil {
				handleErr(err)
			}
			return
		}

		if _, err := w.Write([]byte("Hello!")); err != nil {
			handleErr(err)
		}
	}

	log.Fatal(http.ListenAndServe(":8080",
		s2s2Client.RequireAuthentication(
			http.HandlerFunc(myHandler),
		).RecordMetricsOnly()))
}

// RequireAuthenatication mode will authenticate s2s2 callers that present
// authentication. If they don't, the request will still be processed,
// and s2s2 metrics will be logged to understand the service usage.
//
// Use OptionalRequestSubject to see if the RequestSubject is set to validate
// the authenticated caller.
func ExampleS2S2_RequireAuthentication_serviceAuthz() {
	var s2s2Client *s2s2.S2S2

	myHandler := func(w http.ResponseWriter, r *http.Request) {
		if requestSubject, ok := s2s2.OptionalRequestSubject(r.Context()); ok {
			if !requestSubject.In(
				s2s2Client.NewAuthorizedDistributedIdentity("service-name", "beta"), // distributed identity service
				// There is no good way for callee service to get service id using service name.
				// meanwhile just configure the allow list using service id.
				s2s2Client.NewAuthorizedService("expected-caller-id"), // s2s v0 service
			) {
				// serve 403
			} else {
				// requested subject is the service you are looking for.
				// do service specific authz here.
			}
			if _, err := w.Write([]byte(fmt.Sprintf("Hello %s!", requestSubject.ID()))); err != nil {
				handleErr(err)
			}
			return
		}

		if _, err := w.Write([]byte("Hello!")); err != nil {
			handleErr(err)
		}
	}

	log.Fatal(http.ListenAndServe(":8080",
		s2s2Client.RequireAuthentication(
			http.HandlerFunc(myHandler),
		).RecordMetricsOnly()))
}
