package s2s2

import (
	"context"
	"errors"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2/internal/authorization"
	"code.justin.tv/amzn/TwitchS2S2/internal/opwrap"
	"code.justin.tv/amzn/TwitchS2S2/s2s2/mocks"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/s2s2dicallee"
	"code.justin.tv/video/metrics-middleware/v2/operation"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestS2S2Handler(t *testing.T) {
	reqMatcher := func(req *http.Request) func(*http.Request) bool {
		return func(inReq *http.Request) bool {
			return inReq.Method == req.Method && inReq.URL.Path == req.URL.Path
		}
	}

	t.Run("PassthroughIfAuthorizationNotPresented", func(t *testing.T) {
		t.Run("authorization presented", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			pht := newS2S2HandlerTest(ctrl)
			defer pht.Teardown(t)

			req, err := http.NewRequest("GET", "/derp", nil)
			require.NoError(t, err)
			req.Header.Set("Authorization", "sent")

			pht.Handler.On("ServeHTTP", nil, mock.MatchedBy(reqMatcher(req))).Return().Once()
			pht.S2S2Handler.PassthroughIfAuthorizationNotPresented().ServeHTTP(nil, req)
		})

		t.Run("authorization not presented", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			pht := newS2S2HandlerTest(ctrl)
			defer pht.Teardown(t)

			req, err := http.NewRequest("GET", "/derp", nil)
			require.NoError(t, err)

			pht.NoAuthHandler.On("ServeHTTP", nil, mock.MatchedBy(reqMatcher(req))).Return().Once()
			pht.mockedOpMonitor.On("MonitorOp", mock.Anything, opwrap.ServePassthroughRequest).
				Return(context.Background(), &operation.MonitorPoints{}).Once()

			pht.S2S2Handler.PassthroughIfAuthorizationNotPresented().ServeHTTP(nil, req)
		})
	})

	t.Run("RecordMetricsOnly", func(t *testing.T) {
		t.Run("Authorization Header missing", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			pht := newS2S2HandlerTest(ctrl)
			defer pht.Teardown(t)

			req, err := http.NewRequest("GET", "/derp", nil)
			require.NoError(t, err)
			req.RemoteAddr = "1.2.3.4"
			req.Header.Set("x-forwarded-for", "X-FORWARDED-FOR")
			req.Header.Set("x-forwarded-proto", "X-FORWARDED-PROTO")
			req.Header.Set("x-forwarded-port", "X-FORWARDED-PORT")

			w := httptest.NewRecorder()
			pht.NoAuthHandler.On("ServeHTTP", w, mock.MatchedBy(reqMatcher(req))).Return().Once()
			pht.mockedDICallee.On("ValidateAuthentication", req).Return(nil, s2s2dicallee.ErrNoAuthenticationPresented).Once()
			pht.mockedOpMonitor.On("MonitorOp", mock.Anything, opwrap.S2S2AuthHeaderMissing).
				Return(context.Background(), &operation.MonitorPoints{}).Once()
			pht.mockedLogger.EXPECT().LogAnonymousRequest("1.2.3.4", "X-FORWARDED-FOR", "X-FORWARDED-PROTO", "X-FORWARDED-PORT")

			pht.S2S2Handler.RecordMetricsOnly().ServeHTTP(w, req)
		})

		t.Run("Authorization Header missing x5u", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			pht := newS2S2HandlerTest(ctrl)
			defer pht.Teardown(t)

			req, err := http.NewRequest("GET", "/derp", nil)
			require.NoError(t, err)

			pht.NoAuthHandler.On("ServeHTTP", nil, mock.MatchedBy(reqMatcher(req))).Return().Once()
			pht.mockedDICallee.On("ValidateAuthentication", req).Return(nil, s2s2dicallee.ErrMissingX5UHeader).Once()
			pht.mockedOpMonitor.On("MonitorOp", mock.Anything, opwrap.S2S2AuthHeaderMissingX5U).
				Return(context.Background(), &operation.MonitorPoints{}).Once()

			pht.S2S2Handler.RecordMetricsOnly().ServeHTTP(nil, req)
		})

		t.Run("Authorization Header: invalid x5u, no s2s2 auth header", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			pht := newS2S2HandlerTest(ctrl)
			defer pht.Teardown(t)

			req, err := http.NewRequest("GET", "/derp", nil)
			require.NoError(t, err)

			w := httptest.NewRecorder()

			pht.NoAuthHandler.On("ServeHTTP", w, mock.MatchedBy(reqMatcher(req))).Return().Once()
			pht.mockedDICallee.On("ValidateAuthentication", req).Return(nil, errors.New("some-error")).Once()

			pht.mockedOpMonitor.On("MonitorOp", mock.Anything, opwrap.S2S2AuthHeaderInvalid).
				Return(context.Background(), &operation.MonitorPoints{}).Once()

			pht.S2S2Handler.RecordMetricsOnly().ServeHTTP(w, req)
		})

		t.Run("Authorization Header: invalid x5u, invalid s2s2 auth header", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			pht := newS2S2HandlerTest(ctrl)
			defer pht.Teardown(t)

			req, err := http.NewRequest("GET", "/derp", nil)
			require.NoError(t, err)

			w := httptest.NewRecorder()

			pht.NoAuthHandler.On("ServeHTTP", w, mock.MatchedBy(reqMatcher(req))).Return().Once()
			pht.mockedDICallee.On("ValidateAuthentication", req).Return(nil, errors.New("some-error")).Once()

			pht.mockedOpMonitor.On("MonitorOp", mock.Anything, opwrap.S2S2AuthHeaderInvalid).
				Return(context.Background(), &operation.MonitorPoints{}).Once()

			pht.S2S2Handler.RecordMetricsOnly().ServeHTTP(w, req)
		})

		t.Run("Valid s2s2 DI Authorization Header", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			pht := newS2S2HandlerTest(ctrl)
			defer pht.Teardown(t)

			req, err := http.NewRequest("GET", "/derp", nil)
			require.NoError(t, err)

			req.Header.Set("Authorization", "Bearer something")
			authzSub := &distributedIdentitiesAuthorizedSubject{}
			pht.NoAuthHandler.On("ServeHTTP", nil, mock.MatchedBy(reqMatcher(req))).Run(
				func(args mock.Arguments) {
					req := args.Get(1).(*http.Request)
					ctx := req.Context()
					sub, ok := OptionalRequestSubject(ctx)
					assert.True(t, ok)
					assert.Equal(t, "", sub.TokenID())
				}).
				Return().Once()

			pht.mockedDICallee.On("ValidateAuthentication", req).Return(authzSub, nil).Once()

			pht.S2S2Handler.RecordMetricsOnly().ServeHTTP(nil, req)
		})

		t.Run("Authorization Header: invalid x5u, valid s2s2 auth header", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			pht := newS2S2HandlerTest(ctrl)
			defer pht.Teardown(t)

			req, err := http.NewRequest("GET", "/derp", nil)
			require.NoError(t, err)

			req.Header.Set("Authorization", "Bearer something")
			req.Host = "127.0.0.1"

			subj := authorization.NewSubject("test-subject")
			authzSub := &authorizedSubject{
				tokenID: "tokenID",
				Subject: subj,
			}

			pht.NoAuthHandler.On("ServeHTTP", nil, mock.MatchedBy(reqMatcher(req))).Run(
				func(args mock.Arguments) {
					req := args.Get(1).(*http.Request)
					ctx := req.Context()
					sub, ok := OptionalRequestSubject(ctx)
					assert.True(t, ok)
					assert.Equal(t, authzSub.tokenID, sub.TokenID())
				}).
				Return().Once()

			pht.mockedDICallee.On("ValidateAuthentication", req).Return(nil, errors.New("some-error")).Once()
			pht.mockedAuth.On("Validate", mock.Anything, mock.Anything, mock.Anything).Return(&authorization.Authorization{
				Audience: authorization.NewAudience("http://127.0.0.1"),
				Subject:  subj,
				JWTID:    authzSub.tokenID,
			}, nil).Once()

			pht.S2S2Handler.RecordMetricsOnly().ServeHTTP(nil, req)
		})

	})
}

func newS2S2HandlerTest(ctrl *gomock.Controller) *s2s2HandlerTest {
	handler := new(mocks.Handler)
	noAuthHandler := new(mocks.Handler)
	mockedDICallee := new(mocks.CalleeAPI)
	mockedAuth := new(mocks.AuthorizationsAPI)
	mockedOpMonitor := new(mocks.OpMonitor)
	mockedLogger := mocks.NewMockLoggerAPI(ctrl)
	cfg := &c7s.Config{
		CalleeRealm: "testing",
		Issuer:      "test-issuer",
	}

	return &s2s2HandlerTest{
		Handler:         handler,
		NoAuthHandler:   noAuthHandler,
		mockedDICallee:  mockedDICallee,
		mockedOpMonitor: mockedOpMonitor,
		mockedAuth:      mockedAuth,
		mockedLogger:    mockedLogger,
		S2S2Handler: &s2s2Handler{
			Handler:       handler,
			NoAuthHandler: noAuthHandler,
			Config:        cfg,
			Logger:        mockedLogger,
			OperationStarter: &operation.Starter{
				OpMonitors: []operation.OpMonitor{
					mockedOpMonitor,
				},
			},
			S2S2: &S2S2{
				diCallee:       mockedDICallee,
				authorizations: mockedAuth,
				config:         cfg,
				logger:         newLoggerFromOptions(&Options{}),
				s2s2Logger:     mockedLogger,
			},
		},
	}
}

type s2s2HandlerTest struct {
	Handler         *mocks.Handler
	NoAuthHandler   *mocks.Handler
	S2S2Handler     *s2s2Handler
	mockedDICallee  *mocks.CalleeAPI
	mockedAuth      *mocks.AuthorizationsAPI
	mockedOpMonitor *mocks.OpMonitor
	mockedLogger    *mocks.MockLoggerAPI
}

func (pht *s2s2HandlerTest) Teardown(t *testing.T) {
	pht.Handler.AssertExpectations(t)
	pht.NoAuthHandler.AssertExpectations(t)
	pht.mockedDICallee.AssertExpectations(t)
	pht.mockedAuth.AssertExpectations(t)
	pht.mockedOpMonitor.AssertExpectations(t)
}
