package s2s2

import (
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/internal/token"
	"github.com/stretchr/testify/assert"
)

func TestCacheKey(t *testing.T) {
	t.Run("defaultCacheKey", func(t *testing.T) {
		const host = "host"
		assert.Equal(t, host, defaultCacheKey(token.NewOptions().WithHost(host)))
	})

	t.Run("cache function", func(t *testing.T) {
		now := time.Now()

		fromBaseToken := func(cbs ...func(in *token.Token)) *token.Token {
			rt := &token.Token{Issued: now, ExpiresIn: time.Hour}
			for _, cb := range cbs {
				cb(rt)
			}
			return rt
		}

		staleToken := func(in *token.Token) {
			in.Issued = now.Add(-16 * time.Minute)
		}

		expiredToken := func(in *token.Token) {
			in.ExpiresIn = time.Hour
			in.Issued = now.Add(-(time.Minute + time.Hour))
		}

		withScope := func(scope string) func(*token.Token) {
			return func(in *token.Token) {
				in.Scope = token.ParseScope(scope)
			}
		}

		tcs := []struct {
			Name     string
			Function func(*token.Options, *token.Token) bool
			Options  *token.Options
			Token    *token.Token
			Expected bool
		}{
			{
				Name:     "defaultCacheValueStale/fresh",
				Function: defaultCacheValueStale,
				Options:  token.NewOptions(),
				Token:    fromBaseToken(),
			},
			{
				Name:     "defaultCacheValueStale/stale",
				Function: defaultCacheValueStale,
				Options:  token.NewOptions(),
				Token:    fromBaseToken(staleToken),
				Expected: true,
			},
			{
				Name:     "defaultCacheValueExpired/fresh",
				Function: defaultCacheValueExpired,
				Options:  token.NewOptions(),
				Token:    fromBaseToken(),
			},
			{
				Name:     "defaultCacheValueExpired/expired",
				Function: defaultCacheValueExpired,
				Options:  token.NewOptions(),
				Token:    fromBaseToken(expiredToken),
				Expected: true,
			},
			{
				Name:     "accessTokenStale/fresh with no scope",
				Function: accessTokenStale,
				Options:  token.NewOptions(),
				Token:    fromBaseToken(),
			},
			{
				Name:     "accessTokenStale/fresh with exact scope",
				Function: accessTokenStale,
				Options:  token.NewOptions().WithScope(token.ParseScope("scope1 scope2")),
				Token:    fromBaseToken(withScope("scope1 scope2")),
			},
			{
				Name:     "accessTokenStale/fresh with scope subset",
				Function: accessTokenStale,
				Options:  token.NewOptions().WithScope(token.ParseScope("scope1")),
				Token:    fromBaseToken(withScope("scope1 scope2")),
			},
			{
				Name:     "accessTokenStale/fresh without scope subset",
				Function: accessTokenStale,
				Options:  token.NewOptions().WithScope(token.ParseScope("scope1 scope2")),
				Token:    fromBaseToken(withScope("scope1")),
				Expected: true,
			},
			{
				Name:     "accessTokenStale/stale",
				Function: accessTokenStale,
				Options:  token.NewOptions(),
				Token:    fromBaseToken(staleToken),
				Expected: true,
			},
			{
				Name:     "accessTokenExpired/fresh with no scope",
				Function: accessTokenExpired,
				Options:  token.NewOptions(),
				Token:    fromBaseToken(),
			},
			{
				Name:     "accessTokenExpired/fresh with exact scope",
				Function: accessTokenExpired,
				Options:  token.NewOptions().WithScope(token.ParseScope("scope1 scope2")),
				Token:    fromBaseToken(withScope("scope1 scope2")),
			},
			{
				Name:     "accessTokenExpired/fresh with scope subset",
				Function: accessTokenExpired,
				Options:  token.NewOptions().WithScope(token.ParseScope("scope1")),
				Token:    fromBaseToken(withScope("scope1 scope2")),
			},
			{
				Name:     "accessTokenExpired/fresh without scope subset",
				Function: accessTokenExpired,
				Options:  token.NewOptions().WithScope(token.ParseScope("scope1 scope2")),
				Token:    fromBaseToken(withScope("scope1")),
				Expected: true,
			},
			{
				Name:     "accessTokenExpired/expired",
				Function: accessTokenExpired,
				Options:  token.NewOptions(),
				Token:    fromBaseToken(expiredToken),
				Expected: true,
			},
		}

		for _, tc := range tcs {
			t.Run(tc.Name, func(t *testing.T) {
				assert.Equal(t, tc.Expected, tc.Function(tc.Options, tc.Token))
			})
		}
	})
}
