package s2s2

import (
	"errors"
	"fmt"
	"net/url"
	"strconv"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2/c7s"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/s2s2dicallee"
	"code.justin.tv/video/metrics-middleware/v2/operation"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestNewProvidedClientServiceURI(t *testing.T) {
	tcs := []struct {
		Name          string
		Config        c7s.Config
		Expected      string
		ExpectedError error
	}{
		{
			Name: "only uri",
			Config: c7s.Config{
				ClientServiceURI: "CLIENTSERVICEURI",
			},
			Expected: "CLIENTSERVICEURI",
		},
		{
			Name: "only name",
			Config: c7s.Config{
				ServiceURIByNameBase: "SERVICEURIBYNAMEBASE",
				ClientServiceName:    "CLIENTSERVICENAME",
			},
			Expected: "SERVICEURIBYNAMEBASE/CLIENTSERVICENAME",
		},
		{
			Name: "both should error",
			Config: c7s.Config{
				ClientServiceURI:     "CLIENTSERVICEURI",
				ServiceURIByNameBase: "SERVICEURIBYNAMEBASE",
				ClientServiceName:    "CLIENTSERVICENAME",
			},
			ExpectedError: errors.New("Only one of TwitchS2S2:clientServiceName TwitchS2S2:clientServiceUri can be set"),
		},
		{
			Name:          "neither",
			Config:        c7s.Config{},
			ExpectedError: errClientServiceNameNotSet,
		},
	}

	for _, tc := range tcs {
		t.Run(tc.Name, func(t *testing.T) {
			res, err := newProvidedClientServiceURI(&tc.Config)
			assert.Equal(t, providedClientServiceURI(tc.Expected), res)
			assert.Equal(t, tc.ExpectedError, err)
		})
	}
}

func TestNewServiceOrigins(t *testing.T) {
	tcs := []struct {
		Config   c7s.Config
		Expected []string
	}{
		{
			Config: c7s.Config{
				ServiceOrigins: "https://origin.1",
			},
			Expected: []string{"https://origin.1"},
		},
		{
			Config: c7s.Config{
				ServiceOrigins: "https://origin.1,https://origin.2",
			},
			Expected: []string{"https://origin.1", "https://origin.2"},
		},
		{
			Config: c7s.Config{
				ServiceOrigins: "     https://origin.1      ,       https://origin.2      ",
			},
			Expected: []string{"https://origin.1", "https://origin.2"},
		},
		{
			Config:   c7s.Config{},
			Expected: nil,
		},
	}

	for nTc, tc := range tcs {
		t.Run(strconv.Itoa(nTc), func(t *testing.T) {
			assert.Equal(t, serviceOrigins(tc.Expected), newServiceOrigins(&tc.Config))
		})
	}
}

func TestNewAuthorizedServices(t *testing.T) {
	var empty []s2s2dicallee.Service
	tcs := []struct {
		name                 string
		authorizedServices   string
		expectedAuthServices authorizedServices
	}{
		{
			name:                 "no authorized service",
			authorizedServices:   "",
			expectedAuthServices: authorizedServices(empty),
		},
		{
			name:               "one service authorized",
			authorizedServices: "test/beta",
			expectedAuthServices: authorizedServices{
				{
					Name:  "test",
					Stage: "beta",
				},
			},
		},
		{
			name:               "multiple services authorized",
			authorizedServices: "test/beta, anotherService/prod,",
			expectedAuthServices: authorizedServices{
				{
					Name:  "test",
					Stage: "beta",
				},
				{
					Name:  "anotherService",
					Stage: "prod",
				},
			},
		},
	}

	for _, tc := range tcs {
		t.Run(tc.name, func(t *testing.T) {
			authServices, err := newAuthorizedServices(&Options{
				Config: &c7s.Config{
					AuthorizedDistributedIdentitiesServices: tc.authorizedServices,
				},
			})
			assert.NoError(t, err)
			assert.Equal(t, tc.expectedAuthServices, authServices)
		})
	}

	t.Run("invalid authorized service", func(t *testing.T) {
		_, err := newAuthorizedServices(&Options{
			Config: &c7s.Config{
				AuthorizedDistributedIdentitiesServices: "Invalid format",
			},
		})
		assert.Error(t, err)
	})
}

func TestNewOptions(t *testing.T) {
	serviceOrigins := serviceOrigins{"https://origin"}
	authorizedServices := authorizedServices{s2s2dicallee.Service{Name: "name", Stage: "stage"}}
	certStoreOrigin := certificateStoreOrigin(&url.URL{Scheme: "https", Host: "derp"})
	logger := noopLogger{}
	operationStarter := &operation.Starter{}

	assert.Equal(t, &s2s2dicallee.Options{
		WebOrigins:             serviceOrigins,
		AuthorizedServices:     authorizedServices,
		CertificateStoreOrigin: certStoreOrigin,
		Logger:                 logger,
		OperationStarter:       operationStarter,
	}, newOptions(serviceOrigins, authorizedServices, certStoreOrigin, logger, operationStarter))
}

func TestNewCallee(t *testing.T) {

	certOrigin, err := url.Parse("http://www.twitch.tv")
	require.NoError(t, err)

	t.Run("success noop callee if service origin and authorizedServices not set", func(t *testing.T) {
		opt := &s2s2dicallee.Options{
			CertificateStoreOrigin: certOrigin,
		}
		diCallee, err := newCallee(opt, &noopLogger{})
		assert.NoError(t, err)
		assert.IsType(t, &noopS2S2DICallee{}, diCallee)
	})

	t.Run("success noop callee if service origin is not set", func(t *testing.T) {
		opt := &s2s2dicallee.Options{
			AuthorizedServices: []s2s2dicallee.Service{
				{Name: "test-name", Stage: "beta"},
			},
			CertificateStoreOrigin: certOrigin,
		}
		diCallee, err := newCallee(opt, &noopLogger{})
		assert.NoError(t, err)
		assert.IsType(t, &noopS2S2DICallee{}, diCallee)
	})

}

func TestNewS2S2Logger(t *testing.T) {
	t.Run("success", func(t *testing.T) {
		_, err := newS2S2Logger(&noopLogger{}, &c7s.Config{
			LogAnonymousRequestRateLimit: time.Minute,
		})
		require.NoError(t, err)
	})

	t.Run("error", func(t *testing.T) {
		_, err := newS2S2Logger(&noopLogger{}, &c7s.Config{
			LogAnonymousRequestRateLimit: -1 * time.Second,
		})
		assert.Equal(t, fmt.Errorf("duration configuration must be greater than 0 - got %v", -1*time.Second), err)
	})
}

func TestDefaultedDuration(t *testing.T) {
	const defaultValue = time.Second

	tcs := []struct {
		Value         time.Duration
		Expected      time.Duration
		ExpectedError error
	}{
		{
			Value:    time.Minute,
			Expected: time.Minute,
		},
		{
			Expected: defaultValue,
		},
		{
			Value:         -1 * time.Second,
			ExpectedError: fmt.Errorf("duration configuration must be greater than 0 - got %v", -1*time.Second),
		},
	}

	for nTc, tc := range tcs {
		t.Run(strconv.Itoa(nTc), func(t *testing.T) {
			res, err := defaultedDuration(tc.Value, defaultValue)
			assert.Equal(t, tc.Expected, res)
			assert.Equal(t, tc.ExpectedError, err)
		})
	}
}
