package s2s2

import (
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestWWWAuthenticateChallenge(t *testing.T) {
	t.Run("parseWWWAuthenticateChallenge", func(t *testing.T) {
		tcs := []struct {
			Name        string
			Value       string
			Expected    *wwwAuthenticateChallenge
			ExpectedErr error
		}{
			{
				Name:  "only challenge type",
				Value: "Bearer",
				Expected: &wwwAuthenticateChallenge{
					challengeType: "Bearer",
					parameters:    map[wwwAuthenticateField]string{},
				},
			},
			{
				Name:  "only one parameter",
				Value: "Bearer realm=\"derp\"",
				Expected: &wwwAuthenticateChallenge{
					challengeType: "Bearer",
					parameters: map[wwwAuthenticateField]string{
						wwwAuthenticateFieldRealm: "derp",
					},
				},
			},
			{
				Name:  "multiple parameters",
				Value: "Bearer realm=\"derp\", scope=\"scopea scopeb\"",
				Expected: &wwwAuthenticateChallenge{
					challengeType: "Bearer",
					parameters: map[wwwAuthenticateField]string{
						wwwAuthenticateFieldRealm: "derp",
						wwwAuthenticateFieldScope: "scopea scopeb",
					},
				},
			},
			{
				Name:        "no header",
				Value:       "",
				Expected:    nil,
				ExpectedErr: errNoWWWAuthenticateChallenge,
			},
			{
				Name:        "invalid param no =",
				Value:       "Bearer realm",
				Expected:    nil,
				ExpectedErr: errInvalidWWWAuthenticateChallenge,
			},
			{
				Name:        "invalid param no first \"",
				Value:       "Bearer realm=value\"",
				Expected:    nil,
				ExpectedErr: errInvalidWWWAuthenticateChallenge,
			},
			{
				Name:        "invalid param no second \"",
				Value:       "Bearer realm=\"value",
				Expected:    nil,
				ExpectedErr: errInvalidWWWAuthenticateChallenge,
			},
		}

		for _, tc := range tcs {
			t.Run(tc.Name, func(t *testing.T) {
				req := &http.Response{Header: make(http.Header)}
				req.Header.Set("WWW-Authenticate", tc.Value)

				val, err := parseWWWAuthenticateChallenge(req)
				assert.Equal(t, tc.ExpectedErr, err)
				assert.Equal(t, tc.Expected, val)
			})
		}
	})

	t.Run("Realm", func(t *testing.T) {
		const realm = "myrealm"

		c := newWWWAuthenticateChallenge()
		c.SetRealm(realm)
		assert.Equal(t, realm, c.Realm())
	})

	t.Run("issuer", func(t *testing.T) {
		const issuer = "myissuer"

		c := newWWWAuthenticateChallenge()
		c.SetIssuer(issuer)
		assert.Equal(t, issuer, c.Issuer())
	})

	t.Run("Scopes", func(t *testing.T) {
		scopes := []string{"a", "b", "c"}

		c := newWWWAuthenticateChallenge()
		c.SetScopes(scopes)
		assert.Equal(t, scopes, c.Scopes())
	})

	t.Run("Error", func(t *testing.T) {
		const errorType = "errortype"
		const errorDescription = "errorDescription"

		c := newWWWAuthenticateChallenge()
		c.SetError(errorType, errorDescription)
		assert.Equal(t, authorizationErrorType(errorType), c.ErrorType())
		assert.Equal(t, errorDescription, c.ErrorDescription())
	})
}
