package cachedvalidation

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/validation"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/s2s2dicallee/mocks"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestCachedValidationsValidate(t *testing.T) {
	const token = "my.jwt.token."

	ctx := context.Background()

	t.Run("success from cache", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedValidationsTest(ctrl, 10)

		val := &validation.Validation{
			Expiration: time.Now().Add(time.Hour),
			NotBefore:  time.Now().Add(-time.Hour),
		}
		test.MockValidationsAPI.EXPECT().Validate(ctx, []byte(token)).Return(val, nil)

		res, err := test.CachedValidations.Validate(ctx, []byte(token))
		require.NoError(t, err)
		assert.Equal(t, val, res)

		assert.Len(t, test.CachedValidations.cache, 1)
		assert.Equal(t, 1, test.CachedValidations.evictList.Len())

		res, err = test.CachedValidations.Validate(ctx, []byte(token))
		require.NoError(t, err)
		assert.Equal(t, val, res)

		assert.Len(t, test.CachedValidations.cache, 1)
		assert.Equal(t, 1, test.CachedValidations.evictList.Len())
	})

	t.Run("success from cache but popping", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedValidationsTest(ctrl, 1)

		val1 := &validation.Validation{
			Expiration: time.Now().Add(time.Hour),
			NotBefore:  time.Now().Add(-time.Hour),
		}
		val2 := &validation.Validation{
			Expiration: time.Now().Add(time.Hour),
			NotBefore:  time.Now().Add(-time.Hour),
		}
		test.MockValidationsAPI.EXPECT().Validate(ctx, []byte("token1")).Return(val1, nil)

		res, err := test.CachedValidations.Validate(ctx, []byte("token1"))
		require.NoError(t, err)
		assert.Equal(t, val1, res)

		assert.Len(t, test.CachedValidations.cache, 1)
		assert.Equal(t, 1, test.CachedValidations.evictList.Len())

		for n := 0; n < 100; n++ {
			test.MockValidationsAPI.EXPECT().Validate(ctx, []byte("token2")).Return(val2, nil)
			res, err := test.CachedValidations.Validate(ctx, []byte("token2"))
			require.NoError(t, err)
			assert.Equal(t, val2, res)

			assert.Len(t, test.CachedValidations.cache, 1)
			assert.Equal(t, 1, test.CachedValidations.evictList.Len())
		}
	})

	t.Run("error from cache", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedValidationsTest(ctrl, 10)

		myErr := errors.New("myerr")
		test.MockValidationsAPI.EXPECT().Validate(ctx, []byte(token)).Return(nil, myErr)

		_, err := test.CachedValidations.Validate(ctx, []byte(token))
		assert.Equal(t, myErr, err)
	})

	t.Run("error expired from cache", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedValidationsTest(ctrl, 10)

		val := &validation.Validation{
			// expired token
			Expiration: time.Now().Add(-time.Hour),
			NotBefore:  time.Now().Add(-time.Hour),
		}
		test.MockValidationsAPI.EXPECT().Validate(ctx, []byte(token)).Return(val, nil)

		res, err := test.CachedValidations.Validate(ctx, []byte(token))
		require.NoError(t, err)
		assert.Equal(t, val, res)

		_, err = test.CachedValidations.Validate(ctx, []byte(token))
		assert.Contains(t, err.Error(), "after")
	})
}

type cachedValidationsTest struct {
	*CachedValidations
	*mocks.MockValidationsAPI
}

func newCachedValidationsTest(ctrl *gomock.Controller, maxLen int) *cachedValidationsTest {
	mockValidationsAPI := mocks.NewMockValidationsAPI(ctrl)
	return &cachedValidationsTest{
		CachedValidations:  New(mockValidationsAPI, maxLen),
		MockValidationsAPI: mockValidationsAPI,
	}
}
