package s2s2dicallee

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/s2s2err"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/service"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/validation"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/s2s2dicallee/mocks"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

//go:generate mockgen -package mocks -destination ./mocks/validationiface.go code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/validation/validationiface ValidationsAPI
//go:generate mockgen -package mocks -destination ./mocks/certiface.go code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/cert/certiface CertificatesAPI
//go:generate mockgen -package mocks -destination ./mocks/logutil.go code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/logutil/logutiliface LoggerAPI
//go:generate mockgen -package mocks -destination ./mocks/http.go net/http RoundTripper

func TestCalleeValidateAuthentication(t *testing.T) {
	const serviceDomain = "serviceDomain"
	const serviceName = "serviceName"
	const serviceStage = "serviceStage"
	const token = "token"

	ctx := context.Background()

	subject := func() service.Service {
		return service.Service{
			Domain: serviceDomain,
			Name:   serviceName,
			Stage:  serviceStage,
		}
	}

	httpRequest := func(authorizationHeaderType, token string) *http.Request {
		return (&http.Request{Header: map[string][]string{
			authorizationHeader: {
				fmt.Sprintf("%s %s", authorizationHeaderType, token),
			},
		}}).WithContext(ctx)
	}

	t.Run("success", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)

		test.MockValidationsAPI.EXPECT().Validate(ctx, []byte(token)).Return(&validation.Validation{
			Subject: subject(),
		}, nil)

		res, err := test.ValidateAuthentication(httpRequest(authorizationHeaderTypeBearer, token))
		require.NoError(t, err)
		assert.Equal(t, &authenticatedSubject{service: subject()}, res)
	})

	t.Run("no header", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)

		_, err := test.ValidateAuthentication(&http.Request{Header: map[string][]string{}})
		assert.Equal(t, ErrNoAuthenticationPresented, err)
	})

	t.Run("not enough parts in authorization header", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)

		_, err := test.ValidateAuthentication(&http.Request{Header: map[string][]string{
			authorizationHeader: {
				"Bearer",
			},
		}})
		assert.Equal(t, &ErrInvalidAuthorizationHeader{Value: "Bearer"}, err)
	})

	t.Run("wrong header type", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)

		_, err := test.ValidateAuthentication(httpRequest("derp", token))
		assert.Equal(t, &ErrInvalidAuthorizationHeader{Value: "derp " + token}, err)
	})

	t.Run("Validate error", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)

		myErr := errors.New("myerr")
		test.MockValidationsAPI.EXPECT().Validate(ctx, []byte(token)).Return(nil, myErr)

		_, err := test.ValidateAuthentication(httpRequest(authorizationHeaderTypeBearer, token))
		assert.Equal(t, myErr, err)
	})

	t.Run("error x5u not in cache", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)

		s2s2Err := s2s2err.NewError(s2s2err.CodeX5UNotInCache, errors.New("SomeError"))
		test.MockValidationsAPI.EXPECT().Validate(ctx, []byte(token)).Return(nil, s2s2Err)

		_, err := test.ValidateAuthentication(httpRequest(authorizationHeaderTypeBearer, token))
		assert.Equal(t, s2s2Err, err)

	})
}

func TestCalleeHardRefreshCache(t *testing.T) {
	ctx := context.Background()

	t.Run("success", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)
		test.MockCertificatesAPI.EXPECT().Refresh(ctx).Return(nil)

		require.NoError(t, test.HardRefreshCache(ctx))
	})

	t.Run("cancelled", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		ctx, cancel := context.WithCancel(ctx)
		cancel()

		test := newCalleeTest(ctrl)
		assert.Error(t, test.HardRefreshCache(ctx))
	})

	t.Run("error", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		myErr := errors.New("myerr")
		test := newCalleeTest(ctrl)
		test.MockCertificatesAPI.EXPECT().Refresh(ctx).Return(myErr)

		assert.Equal(t, myErr, test.HardRefreshCache(ctx))
	})
}

type calleeTest struct {
	*Callee
	*mocks.MockValidationsAPI
	*mocks.MockCertificatesAPI
	*mocks.MockLoggerAPI
}

func newCalleeTest(ctrl *gomock.Controller) *calleeTest {
	mockValidationsAPI := mocks.NewMockValidationsAPI(ctrl)
	mockCertificatesAPI := mocks.NewMockCertificatesAPI(ctrl)
	mockLoggerAPI := mocks.NewMockLoggerAPI(ctrl)
	return &calleeTest{
		Callee: &Callee{
			validations:        mockValidationsAPI,
			certificates:       mockCertificatesAPI,
			refreshRateLimiter: time.NewTicker(time.Millisecond),
			logger:             mockLoggerAPI,
		},
		MockCertificatesAPI: mockCertificatesAPI,
		MockValidationsAPI:  mockValidationsAPI,
		MockLoggerAPI:       mockLoggerAPI,
	}
}
