package s2s2dicallee

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/cert"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/service"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCallee/internal/validation"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestHandlerServeHTTP(t *testing.T) {
	const serviceDomain = "serviceDomain"
	const serviceName = "serviceName"
	const serviceStage = "serviceStage"
	const token = "token"
	ctx := context.Background()
	subject := func() service.Service {
		return service.Service{
			Domain: serviceDomain,
			Name:   serviceName,
			Stage:  serviceStage,
		}
	}
	client := &http.Client{}

	httpRequest := func(t *testing.T, url, authorizationHeaderType, token string) *http.Request {
		req, err := http.NewRequest(http.MethodGet, url, nil)
		require.NoError(t, err)
		req.Header.Set(authorizationHeader, fmt.Sprintf("%s %s", authorizationHeaderType, token))
		return req.WithContext(ctx)
	}

	assertBody := func(t *testing.T, res *http.Response, expected string) {
		bs, err := ioutil.ReadAll(res.Body)
		require.NoError(t, err)
		assert.Equal(t, expected, string(bs))
	}

	assertErrorBody := func(t *testing.T, res *http.Response, expected jsonError) {
		var received jsonError
		require.NoError(t, json.NewDecoder(res.Body).Decode(&received))
		assert.Equal(t, expected, received)
	}

	t.Run("success with authorization", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)
		server := httptest.NewServer(test.Callee.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			_, err := w.Write([]byte("ok"))
			require.NoError(t, err)

			subject, ok := RequestSubject(r.Context())
			require.True(t, ok)
			assert.Equal(t, subject, subject)
		})))
		defer server.Close()

		test.MockValidationsAPI.EXPECT().
			Validate(gomock.Any(), []byte(token)).
			Return(&validation.Validation{
				Subject: subject(),
			}, nil)

		res, err := client.Do(httpRequest(t, server.URL, authorizationHeaderTypeBearer, token))
		require.NoError(t, err)
		defer res.Body.Close()
		assert.Equal(t, http.StatusOK, res.StatusCode)
		assertBody(t, res, "ok")
	})

	t.Run("validation error", func(t *testing.T) {
		t.Run("ErrNoAuthenticationPresented", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			test := newCalleeTest(ctrl)
			server := httptest.NewServer(test.Callee.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				_, err := w.Write([]byte("ok"))
				require.NoError(t, err)
			})))
			defer server.Close()

			test.MockLoggerAPI.EXPECT().LogAnonymousRequest(gomock.Any())

			req, err := http.NewRequest(http.MethodGet, server.URL, nil)
			require.NoError(t, err)

			res, err := client.Do(req)
			require.NoError(t, err)
			defer res.Body.Close()
			assert.Equal(t, http.StatusUnauthorized, res.StatusCode)
			assertErrorBody(t, res, jsonError{
				Code: "unauthenticated",
				Msg:  "no Authorization header presented",
			})
		})

		t.Run("ErrX5UNotInCache", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			test := newCalleeTest(ctrl)
			server := httptest.NewServer(test.Callee.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				_, err := w.Write([]byte("ok"))
				require.NoError(t, err)
			})))
			defer server.Close()

			test.MockLoggerAPI.EXPECT().LogUncachedX5U("myx5u")
			test.MockValidationsAPI.EXPECT().
				Validate(gomock.Any(), []byte(token)).
				Return(nil, &cert.ErrX5UNotInCache{X5U: "myx5u"})

			res, err := client.Do(httpRequest(t, server.URL, authorizationHeaderTypeBearer, token))
			require.NoError(t, err)
			defer res.Body.Close()
			assert.Equal(t, http.StatusForbidden, res.StatusCode)
			assertErrorBody(t, res, jsonError{
				Code: "permission_denied",
				Msg:  "x5u not in cache: myx5u",
			})
		})

		t.Run("unknown", func(t *testing.T) {
			ctrl := gomock.NewController(t)
			defer ctrl.Finish()

			test := newCalleeTest(ctrl)
			server := httptest.NewServer(test.Callee.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				_, err := w.Write([]byte("ok"))
				require.NoError(t, err)
			})))
			defer server.Close()

			myErr := errors.New("myerr")
			test.MockLoggerAPI.EXPECT().LogUnknownError(myErr)
			test.MockValidationsAPI.EXPECT().
				Validate(gomock.Any(), []byte(token)).
				Return(nil, myErr)

			res, err := client.Do(httpRequest(t, server.URL, authorizationHeaderTypeBearer, token))
			require.NoError(t, err)
			defer res.Body.Close()
			assert.Equal(t, http.StatusInternalServerError, res.StatusCode)
			assertErrorBody(t, res, jsonError{
				Code: "internal",
				Msg:  "myerr",
			})
		})
	})
}

func TestHandlerRecordMetricsOnly(t *testing.T) {
	const serviceDomain = "serviceDomain"
	const serviceName = "serviceName"
	const serviceStage = "serviceStage"
	const token = "token"
	ctx := context.Background()
	subject := func() service.Service {
		return service.Service{
			Domain: serviceDomain,
			Name:   serviceName,
			Stage:  serviceStage,
		}
	}
	client := &http.Client{}

	httpRequest := func(t *testing.T, url, authorizationHeaderType, token string) *http.Request {
		req, err := http.NewRequest(http.MethodGet, url, nil)
		require.NoError(t, err)
		req.Header.Set(authorizationHeader, fmt.Sprintf("%s %s", authorizationHeaderType, token))
		return req.WithContext(ctx)
	}

	assertBody := func(t *testing.T, res *http.Response, expected string) {
		bs, err := ioutil.ReadAll(res.Body)
		require.NoError(t, err)
		assert.Equal(t, expected, string(bs))
	}

	t.Run("success with authorization", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)
		server := httptest.NewServer(test.Callee.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			_, err := w.Write([]byte("ok"))
			require.NoError(t, err)

			subject, ok := RequestSubject(r.Context())
			require.True(t, ok)
			assert.Equal(t, subject, subject)
		})).RecordMetricsOnly())
		defer server.Close()

		test.MockValidationsAPI.EXPECT().
			Validate(gomock.Any(), []byte(token)).
			Return(&validation.Validation{
				Subject: subject(),
			}, nil)

		res, err := client.Do(httpRequest(t, server.URL, authorizationHeaderTypeBearer, token))
		require.NoError(t, err)
		defer res.Body.Close()
		assert.Equal(t, http.StatusOK, res.StatusCode)
		assertBody(t, res, "ok")
	})

	t.Run("error should not error out", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCalleeTest(ctrl)
		server := httptest.NewServer(test.Callee.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			_, err := w.Write([]byte("ok"))
			require.NoError(t, err)
		})).RecordMetricsOnly())
		defer server.Close()

		myErr := errors.New("myerr")
		test.MockLoggerAPI.EXPECT().LogUnknownError(myErr)
		test.MockValidationsAPI.EXPECT().
			Validate(gomock.Any(), []byte(token)).
			Return(nil, myErr)

		res, err := client.Do(httpRequest(t, server.URL, authorizationHeaderTypeBearer, token))
		require.NoError(t, err)
		defer res.Body.Close()
		assert.Equal(t, http.StatusOK, res.StatusCode)
		assertBody(t, res, "ok")
	})
}
