package algorithm

import (
	"crypto/ecdsa"
	"encoding/asn1"
	"errors"
	"math/big"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/s2s2err"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/aws/aws-sdk-go/service/kms/kmsiface"
)

// KMSAlgorithm implements jwt.Algorithm
type KMSAlgorithm struct {
	KMS       kmsiface.KMSAPI
	AliasName string
	PublicKey *ecdsa.PublicKey
}

// Size implements jwt.Algorithm
func (a *KMSAlgorithm) Size() int { return 96 }

// Name implements jwt.Algorithm
func (a *KMSAlgorithm) Name() string { return "ES384" }

// Sign implements jwt.Algorithm
func (a *KMSAlgorithm) Sign(value []byte) (signature []byte, err error) {
	out, err := a.KMS.Sign(&kms.SignInput{
		KeyId:            aws.String(a.AliasName),
		Message:          value,
		MessageType:      aws.String(kms.MessageTypeRaw),
		SigningAlgorithm: aws.String(kms.SigningAlgorithmSpecEcdsaSha384),
	})
	if err != nil {
		return nil, s2s2err.NewError(s2s2err.CodeKMSSignError, err)
	}

	// https://tools.ietf.org/html/rfc3279#section-2.2.3
	var ecdsaSignature struct {
		R *big.Int
		S *big.Int
	}

	if _, err := asn1.Unmarshal(out.Signature, &ecdsaSignature); err != nil {
		return nil, err
	}

	jws := make([]byte, a.Size())
	rBytes := ecdsaSignature.R.Bytes()
	sBytes := ecdsaSignature.S.Bytes()

	copy(jws[a.Size()/2-len(rBytes):a.Size()], rBytes)
	copy(jws[a.Size()-len(sBytes):], sBytes)

	return jws, nil
}

// Validate implements jwt.Algorithm
func (a *KMSAlgorithm) Validate(value, signature []byte) error {
	return errors.New("Validate not implemented")
}
