package algorithm

import (
	"encoding/asn1"
	"errors"
	"math/big"
	"testing"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/mocks/kmsapimock"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
)

func TestSize(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	a := newAlgorithmTest(t, ctrl)
	size := a.kmsAlgorithm.Size()
	expectedSize := 96
	assert.Equal(t, expectedSize, size)
}

func TestName(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	a := newAlgorithmTest(t, ctrl)
	name := a.kmsAlgorithm.Name()
	expectedName := "ES384"
	assert.Equal(t, expectedName, name)
}

func TestSign(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	a := newAlgorithmTest(t, ctrl)
	t.Run("Success", func(t *testing.T) {
		signature, err := dummySig(t)
		assert.NoError(t, err)
		a.KMS.EXPECT().
			Sign(gomock.Any()).
			Return(&kms.SignOutput{
				KeyId:            aws.String("alias/twitch/testService/beta"),
				Signature:        signature,
				SigningAlgorithm: aws.String(kms.SigningAlgorithmSpecEcdsaSha384),
			}, nil)
		signature, err = a.kmsAlgorithm.Sign([]byte("test"))
		assert.NoError(t, err)
		expectedSig := []byte{0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2}
		assert.Equal(t, expectedSig, signature)
	})
	t.Run("Error on KMS sign", func(t *testing.T) {
		a.KMS.EXPECT().
			Sign(gomock.Any()).
			Return(&kms.SignOutput{}, errors.New("Error on calling Sign from KMS"))
		_, err := a.kmsAlgorithm.Sign([]byte("test"))
		assert.Error(t, err)
	})
	t.Run("Error on asn1 unmarshal", func(t *testing.T) {
		a.KMS.EXPECT().
			Sign(gomock.Any()).
			Return(&kms.SignOutput{}, nil)
		_, err := a.kmsAlgorithm.Sign([]byte("test"))
		assert.Error(t, err)
	})
}

func TestValidate(t *testing.T) {
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	a := newAlgorithmTest(t, ctrl)
	t.Run("Error", func(t *testing.T) {
		err := a.kmsAlgorithm.Validate(nil, nil)
		assert.Error(t, err)
	})
}

func dummySig(t *testing.T) ([]byte, error) {
	var ecdsaSignature struct {
		R *big.Int
		S *big.Int
	}
	ecdsaSignature.R = big.NewInt(1)
	ecdsaSignature.S = big.NewInt(2)
	signature, err := asn1.Marshal(ecdsaSignature)
	if err != nil {
		return nil, err
	}
	return signature, nil
}

func newAlgorithmTest(t *testing.T, ctrl *gomock.Controller) *algorithmTest {
	kmsMock := kmsapimock.NewMockKMSAPI(ctrl)
	return &algorithmTest{
		kmsAlgorithm: &KMSAlgorithm{
			KMS:       kmsMock,
			AliasName: "alias/twitch/testService/beta",
			PublicKey: nil,
		},
		KMS: kmsMock,
	}
}

type algorithmTest struct {
	kmsAlgorithm *KMSAlgorithm
	KMS          *kmsapimock.MockKMSAPI
}
