package authentication

import (
	"context"
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/algorithm"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/s2s2err"
	"code.justin.tv/sse/jwt"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/aws/aws-sdk-go/service/kms/kmsiface"
)

const notBefore = -1 * time.Hour

// Authentications authenticates a service to another with no caching.
type Authentications struct {
	IdentityOrigin string
	ServiceDomain  string
	ServiceName    string
	Stage          string
	KMS            kmsiface.KMSAPI
	Expiration     time.Duration
}

// Authenticate authenticates a request
func (a Authentications) Authenticate(ctx context.Context, audienceHost string) (_ []byte, err error) {
	aliasName := fmt.Sprintf("alias/twitch/%s/%s", a.ServiceName, a.Stage)

	describeKeyOutput, err := a.KMS.DescribeKeyWithContext(ctx, &kms.DescribeKeyInput{
		KeyId: aws.String(aliasName),
	})
	if err != nil {
		return nil, s2s2err.NewError(s2s2err.CodeKMSDescribeKeyError, err)
	}

	x509URL := fmt.Sprintf("%s/%s/%s/%s/%s.json",
		a.IdentityOrigin,
		a.ServiceDomain,
		a.ServiceName,
		a.Stage,
		*describeKeyOutput.KeyMetadata.KeyId)

	alg := &algorithm.KMSAlgorithm{
		KMS:       a.KMS,
		AliasName: aliasName,
	}

	bs, err := jwt.Encode(
		struct {
			Algorithm           string `json:"alg"`
			Type                string `json:"typ"`
			X509CertificatesURL string `json:"x5u"`
		}{
			Algorithm:           alg.Name(),
			Type:                "JWT",
			X509CertificatesURL: x509URL,
		},
		struct {
			Audience   string `json:"aud"`
			NotBefore  int64  `json:"nbf"`
			Expiration int64  `json:"exp"`
		}{
			Audience:   audienceHost,
			NotBefore:  time.Now().Add(notBefore).Unix(),
			Expiration: time.Now().Add(a.Expiration).Unix(),
		},
		alg,
	)
	if err != nil {
		return nil, err
	}
	return bs, nil
}
