package authentication

import (
	"context"
	"encoding/asn1"
	"encoding/base64"
	"errors"
	"math/big"
	"strings"
	"testing"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/mocks/kmsapimock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/golang/mock/gomock"
)

func TestAuthenticate(t *testing.T) {
	ctx := context.Background()
	ctrl := gomock.NewController(t)
	defer ctrl.Finish()
	kmsKeyID := "12345"
	audienceHost := "testAudience"

	signature, err := dummySig(t)
	assert.NoError(t, err)
	a := newAuthenticationTest(t, ctrl)

	t.Run("Success", func(t *testing.T) {
		mockedKeyMetaData := &kms.KeyMetadata{
			KeyId: aws.String(kmsKeyID),
		}
		a.KMS.EXPECT().
			DescribeKeyWithContext(gomock.Any(), gomock.Any()).
			Return(&kms.DescribeKeyOutput{
				KeyMetadata: mockedKeyMetaData,
			}, nil)
		a.KMS.EXPECT().
			Sign(gomock.Any()).
			Return(&kms.SignOutput{
				KeyId:            aws.String(kmsKeyID),
				Signature:        signature,
				SigningAlgorithm: aws.String(kms.SigningAlgorithmSpecEcdsaSha256),
			}, nil)
		bs, err := a.authentications.Authenticate(ctx, audienceHost)
		assert.NoError(t, err)

		b64Decoded := func(t *testing.T, s string) string {
			out, err := base64.RawStdEncoding.DecodeString(s)
			require.NoError(t, err)
			return string(out)
		}

		resultSplit := strings.Split(string(bs), ".")
		t.Logf("token: %s", string(bs))
		headerResult := resultSplit[0]
		t.Logf("header: %s", b64Decoded(t, resultSplit[0]))
		claimsResult := resultSplit[1]
		t.Logf("claims: %s", b64Decoded(t, resultSplit[1]))
		expectedHeader := []byte("{\"alg\":\"ES384\",\"typ\":\"JWT\",\"x5u\":\"https://prod.s2s2identities.twitch.a2z.com/twitch/testService/beta/12345.json\"}")
		t.Logf("expected: %s", string(expectedHeader))
		assert.Equal(t, base64.RawStdEncoding.EncodeToString(expectedHeader), headerResult)
		// Checking the claims for only the Audience field as the others will be based on current time
		expectedClaims := []byte("{\"aud\":\"testAudience\"")
		assert.Equal(t, base64.StdEncoding.EncodeToString(expectedClaims), claimsResult[:28])
	})
	t.Run("ErrorOnDescribeKey", func(t *testing.T) {
		a.KMS.EXPECT().
			DescribeKeyWithContext(gomock.Any(), gomock.Any()).
			Return(&kms.DescribeKeyOutput{}, errors.New("Error on describe key"))
		_, err := a.authentications.Authenticate(ctx, audienceHost)
		assert.Error(t, err)
	})
	t.Run("ErrorOnJWTEncode", func(t *testing.T) {
		mockedKeyMetaData := &kms.KeyMetadata{
			KeyId: aws.String(kmsKeyID),
		}
		a.KMS.EXPECT().
			DescribeKeyWithContext(gomock.Any(), gomock.Any()).
			Return(&kms.DescribeKeyOutput{
				KeyMetadata: mockedKeyMetaData,
			}, nil)
		a.KMS.EXPECT().
			Sign(gomock.Any()).
			Return(&kms.SignOutput{}, errors.New("Error on JWT encode"))
		_, err := a.authentications.Authenticate(ctx, audienceHost)
		assert.Error(t, err)
	})
}

func dummySig(t *testing.T) ([]byte, error) {
	var ecdsaSignature struct {
		R *big.Int
		S *big.Int
	}
	ecdsaSignature.R = big.NewInt(1)
	ecdsaSignature.S = big.NewInt(2)
	signature, err := asn1.Marshal(ecdsaSignature)
	if err != nil {
		return nil, err
	}
	return signature, nil
}

func newAuthenticationTest(t *testing.T, ctrl *gomock.Controller) *authenticationTest {
	kmsMock := kmsapimock.NewMockKMSAPI(ctrl)
	return &authenticationTest{
		authentications: &Authentications{
			IdentityOrigin: "https://prod.s2s2identities.twitch.a2z.com",
			ServiceName:    "testService",
			Stage:          "beta",
			ServiceDomain:  "twitch",
			KMS:            kmsMock,
		},
		KMS: kmsMock,
	}
}

type authenticationTest struct {
	authentications *Authentications
	KMS             *kmsapimock.MockKMSAPI
}
