package cachedauthentication

import (
	"context"
	"errors"
	"sync"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/s2s2err"
	"code.justin.tv/video/metrics-middleware/v2/operation"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type returnOnceAuthentications struct {
	Token []byte
	sync.Once
}

func (m *returnOnceAuthentications) Authenticate(context.Context, string) ([]byte, error) {
	var ret []byte
	err := errors.New("myerr")
	m.Once.Do(func() {
		ret = m.Token
		err = nil
	})
	return ret, err
}

func BenchmarkCachedAuthenticationsFromCache(b *testing.B) {
	const audienceHost = "https://audience"
	const token = "b"

	ctx := context.Background()

	b.Run("from cache", func(b *testing.B) {
		ca := New(&returnOnceAuthentications{Token: []byte(token)}, time.Hour, 24*time.Hour, time.NewTicker(time.Minute), new(s2s2err.Logger), &operation.Starter{})

		b.RunParallel(func(pb *testing.PB) {
			for pb.Next() {
				res, err := ca.Authenticate(ctx, audienceHost)
				require.NoError(b, err)
				require.Equal(b, []byte(token), res)
			}
		})
	})

	b.Run("from cache with hard refreshes", func(b *testing.B) {
		ca := New(&returnOnceAuthentications{Token: []byte(token)}, time.Hour, 24*time.Hour, time.NewTicker(time.Minute), new(s2s2err.Logger), &operation.Starter{})

		b.RunParallel(func(pb *testing.PB) {
			for pb.Next() {
				res, err := ca.Authenticate(ctx, audienceHost)
				require.NoError(b, err)
				require.Equal(b, []byte(token), res)
			}
		})
	})

	b.Run("stale failure", func(b *testing.B) {
		ca := New(&returnOnceAuthentications{Token: []byte(token)}, time.Hour, 24*time.Hour, time.NewTicker(time.Minute), new(s2s2err.Logger), &operation.Starter{})

		res, err := ca.Authenticate(ctx, audienceHost)
		require.NoError(b, err)
		assert.Equal(b, []byte(token), res)

		// simulate stale
		ca.cache[audienceHost].CreationTime = time.Now().Add(-2 * time.Hour)

		b.RunParallel(func(pb *testing.PB) {
			for pb.Next() {
				res, err := ca.Authenticate(ctx, audienceHost)
				require.NoError(b, err)
				require.Equal(b, []byte(token), res)
			}
		})
	})
}
