package cachedauthentication

import (
	"context"
	"sync"
	"time"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/authentication/authenticationiface"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/cacheitem"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/opwrap"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/s2s2err"
	"code.justin.tv/video/metrics-middleware/v2/operation"
)

// New returns a new Authentications client
func New(
	authentications authenticationiface.AuthenticationsAPI,
	staleTimeout time.Duration,
	expirationTimeout time.Duration,
	refreshRateLimiter *time.Ticker,
	logger *s2s2err.Logger,
	operationStarter *operation.Starter,
) *CachedAuthentications {
	return &CachedAuthentications{
		authentications:    authentications,
		staleTimeout:       staleTimeout,
		expirationTimeout:  expirationTimeout,
		cache:              make(map[string]*cacheitem.CacheItem),
		refreshRateLimiter: refreshRateLimiter,
		logger:             logger,
		operationStarter:   operationStarter,
	}
}

// CachedAuthentications is a cached implementation of
// authentication.Authentications
type CachedAuthentications struct {
	authentications    authenticationiface.AuthenticationsAPI
	staleTimeout       time.Duration
	expirationTimeout  time.Duration
	refreshRateLimiter *time.Ticker
	logger             *s2s2err.Logger
	operationStarter   *operation.Starter

	cacheLock sync.RWMutex
	cache     map[string]*cacheitem.CacheItem
}

// Authenticate authenticates a request from cache if possible
func (a *CachedAuthentications) Authenticate(ctx context.Context, audienceHost string) ([]byte, error) {
	value := a.fetchFromCache(audienceHost)
	if value == nil {
		value, err := a.refetchAuthenticate(ctx, audienceHost, nil)
		if err != nil {
			return nil, err
		}
		return value.Token, nil
	}

	if !value.IsStale() {
		return value.Token, nil
	}

	refetchedValue, err := a.refetchAuthenticate(ctx, audienceHost, value)
	if err != nil {
		if value.IsExpired() {
			return nil, err
		}

		// error is not handled - log the occurance for debug.
		a.logger.LogError(err)

		return value.Token, nil
	}

	return refetchedValue.Token, nil
}

// HardRefreshCache will hard refresh all tokens in our cache.
func (a *CachedAuthentications) HardRefreshCache(ctx context.Context) error {
	for _, host := range a.hostsInCache() {
		if _, err := a.forceRefetchAuthenticate(ctx, host, nil); err != nil {
			return err
		}
	}
	return nil
}

func (a *CachedAuthentications) hostsInCache() []string {
	a.cacheLock.RLock()
	defer a.cacheLock.RUnlock()

	hosts := make([]string, 0, len(a.cache))
	for host := range a.cache {
		hosts = append(hosts, host)
	}

	return hosts
}

func (a *CachedAuthentications) fetchFromCache(audienceHost string) *cacheitem.CacheItem {
	a.cacheLock.RLock()
	defer a.cacheLock.RUnlock()
	return a.cache[audienceHost]
}

func (a *CachedAuthentications) updateCache(audienceHost string, value *cacheitem.CacheItem) {
	a.cacheLock.Lock()
	defer a.cacheLock.Unlock()
	a.cache[audienceHost] = value
}

func (a *CachedAuthentications) refetchAuthenticate(ctx context.Context, audienceHost string, state *cacheitem.CacheItem) (*cacheitem.CacheItem, error) {
	return a.refetchAuthenticateWithForceRefreshOption(ctx, audienceHost, state, false)
}

func (a *CachedAuthentications) forceRefetchAuthenticate(ctx context.Context, audienceHost string, state *cacheitem.CacheItem) (*cacheitem.CacheItem, error) {
	return a.refetchAuthenticateWithForceRefreshOption(ctx, audienceHost, state, true)
}

func (a *CachedAuthentications) refetchAuthenticateWithForceRefreshOption(
	ctx context.Context,
	audienceHost string,
	state *cacheitem.CacheItem,
	forceRefresh bool,
) (*cacheitem.CacheItem, error) {

	if !forceRefresh {
		if currentValue := a.fetchFromCache(audienceHost); currentValue != state && currentValue != nil {
			// we didn't get the lock first
			return currentValue, nil
		}
	}

	if state != nil && !state.IsExpired() {
		// only refetch up to a rate limiter if we're not expired or nil
		select {
		case <-a.refreshRateLimiter.C:
		default:
			return state, nil
		}
	}

	fetchTime := time.Now()

	token, err := func() (_ []byte, err error) {
		opName := opwrap.GetAuthToken
		if forceRefresh {
			opName = opwrap.GetAuthTokenInBackground
		}

		ctx, op := a.operationStarter.StartOp(ctx, opName)
		defer opwrap.EndWithError(op, err)

		return a.authentications.Authenticate(ctx, audienceHost)
	}()
	if err != nil {
		return nil, err
	}

	value := &cacheitem.CacheItem{
		Token:             token,
		CreationTime:      fetchTime,
		StaleTimeout:      a.staleTimeout,
		ExpirationTimeout: a.expirationTimeout,
	}
	a.updateCache(audienceHost, value)
	return value, nil
}
