package cachedauthentication

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/mocks/authenticationmock"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/s2s2err"
	"code.justin.tv/video/metrics-middleware/v2/operation"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestCachedAuthenticationsAuthenticate(t *testing.T) {
	const staleTimeout = time.Hour
	const expirationTimeout = 420 * time.Hour
	const audienceHost = "https://audience"
	const token = "mytoken"

	ctx := context.Background()

	t.Run("success from fresh cache", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedAuthenticationsTest(ctrl, staleTimeout, expirationTimeout)

		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(token), nil)

		res, err := test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)

		// should fetch from cache this time
		res, err = test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)
	})

	t.Run("success from stale cache", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedAuthenticationsTest(ctrl, staleTimeout, expirationTimeout)

		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(token), nil).
			Times(2)

		res, err := test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)

		// simulate timeout
		test.CachedAuthentications.cache[audienceHost].CreationTime = time.Now().Add(-2 * staleTimeout)
		time.Sleep(10 * time.Millisecond)

		res, err = test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)
	})

	t.Run("success from stale cache but refresh failure", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedAuthenticationsTest(ctrl, staleTimeout, expirationTimeout)

		myErr := errors.New("myerr")
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(token), nil)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return(nil, myErr)

		res, err := test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)

		// simulate timeout
		test.CachedAuthentications.cache[audienceHost].CreationTime = time.Now().Add(-2 * staleTimeout)
		time.Sleep(10 * time.Millisecond)

		res, err = test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)

		// 3rd call should immediately return the cached value
		res, err = test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)
	})

	t.Run("failure on no cache", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedAuthenticationsTest(ctrl, staleTimeout, expirationTimeout)

		myErr := errors.New("myerr")
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return(nil, myErr)

		_, err := test.CachedAuthentications.Authenticate(ctx, audienceHost)
		assert.Equal(t, myErr, err)
	})

	t.Run("failure on expired cache", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedAuthenticationsTest(ctrl, staleTimeout, expirationTimeout)

		myErr := errors.New("myerr")
		//ctx will be updated by operationStarter.StartOp
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(token), nil)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return(nil, myErr)

		res, err := test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)

		// simulate expiration
		test.CachedAuthentications.cache[audienceHost].CreationTime = time.Now().Add(-2 * expirationTimeout)

		_, err = test.CachedAuthentications.Authenticate(ctx, audienceHost)
		assert.Equal(t, myErr, err)
	})
}

func TestCachedAuthenticationsHardRefreshCache(t *testing.T) {
	const staleTimeout = time.Hour
	const expirationTimeout = 420 * time.Hour
	const audienceHost = "https://audience"
	const token = "mytoken"
	const refreshedToken = "myrefreshedtoken"
	ctx := context.Background()

	authenticateHostFirstTime := func(t *testing.T, test *cachedAuthenticationsTest) {
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(token), nil)

		res, err := test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)
	}

	t.Run("success", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedAuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		authenticateHostFirstTime(t, test)

		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(refreshedToken), nil)

		require.NoError(t, test.HardRefreshCache(ctx))

		res, err := test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(refreshedToken), res)
	})

	t.Run("failure", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newCachedAuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		authenticateHostFirstTime(t, test)

		myErr := errors.New("myerr")
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return(nil, myErr)

		assert.Equal(t, myErr, test.HardRefreshCache(ctx))

		res, err := test.CachedAuthentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		assert.Equal(t, []byte(token), res)
	})
}

type cachedAuthenticationsTest struct {
	*CachedAuthentications
	*authenticationmock.MockAuthenticationsAPI
}

func newCachedAuthenticationsTest(
	ctrl *gomock.Controller,
	staleTimeout time.Duration,
	expirationTimeout time.Duration,
) *cachedAuthenticationsTest {
	mockAuthenticationsAPI := authenticationmock.NewMockAuthenticationsAPI(ctrl)
	return &cachedAuthenticationsTest{
		CachedAuthentications: New(
			mockAuthenticationsAPI,
			staleTimeout,
			expirationTimeout,
			time.NewTicker(time.Millisecond),
			new(s2s2err.Logger),
			&operation.Starter{},
		),
		MockAuthenticationsAPI: mockAuthenticationsAPI,
	}
}
