package cachedl2authentication

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/mocks/authenticationmock"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/mocks/dynamodbapimock"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/s2s2err"
	"code.justin.tv/video/metrics-middleware/v2/operation"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/require"
)

func TestCachedL2AuthenticationsAuthenticate(t *testing.T) {
	const staleTimeout = time.Hour
	const expirationTimeout = 12 * time.Hour
	const audienceHost = "https://testaudience"
	const token = "testtoken"
	const freshToken = "freshtoken"

	ctx := context.Background()

	t.Run("Succeess on fetch from L2 cache (not stale)", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Format(RFC3339_NANO_FORMAT))},
		}
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		require.Equal(t, res, []byte(token))
	})

	t.Run("Succeess on fetch from L2 (stale), success on update L2", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(-2 * time.Hour).Format(RFC3339_NANO_FORMAT))},
		}
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(freshToken), nil)
		test.MockDynamoDBAPI.EXPECT().PutItem(gomock.Any()).Return(
			&dynamodb.PutItemOutput{},
			nil,
		)

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		require.Equal(t, res, []byte(freshToken))
	})

	t.Run("Failure on fetch from L2 (unmarshaling), failure on KMS", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {S: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(-2 * time.Hour).Format(RFC3339_NANO_FORMAT))},
		}
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return(nil, errors.New("some err"))

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.EqualError(t, err, "token request failed with error some err")
		require.Empty(t, res)
	})

	t.Run("Success on fetch from L2 (expired), success on update L2", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(-20 * time.Hour).Format(RFC3339_NANO_FORMAT))},
		}
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(freshToken), nil)
		test.MockDynamoDBAPI.EXPECT().PutItem(gomock.Any()).Return(
			&dynamodb.PutItemOutput{},
			nil,
		)

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		require.Equal(t, res, []byte(freshToken))
	})

	t.Run("Failure on fetch from L2 cache and KMS", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		authMsg := "token request failed with error some err"
		myerr := errors.New("some err")
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(&dynamodb.GetItemOutput{Item: nil}, nil)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return(nil, myerr)
		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.EqualError(t, err, authMsg)
		require.Empty(t, res)
	})

	t.Run("Succeess on fetch from L2 (stale), fail on update", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(-2 * time.Hour).Format(RFC3339_NANO_FORMAT))},
		}
		myerr := errors.New("update failure")
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(freshToken), nil)
		test.MockDynamoDBAPI.EXPECT().PutItem(gomock.Any()).Return(
			nil,
			myerr,
		)

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		require.Equal(t, res, []byte(token))
	})

	t.Run("Failure on fetch from L2 cache, Success on fetch from KMS, fail on update L2 (conditional), success fetch L2", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(time.Hour).Format(RFC3339_NANO_FORMAT))},
		}
		fetchErr := errors.New("cache miss")
		updateErr := awserr.New("ConditionalCheckFailedException", "The conditional request failed", nil)
		requestErr := awserr.NewRequestFailure(updateErr, 1, "testrequest")

		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			nil,
			fetchErr,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(freshToken), nil)
		test.MockDynamoDBAPI.EXPECT().PutItem(gomock.Any()).Return(
			nil,
			requestErr,
		)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)
		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		require.Equal(t, res, []byte(token))
	})

	t.Run("Failure on fetch from L2 cache, Success on fetch from KMS, fail on update L2 (non-conditional)", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		fetchErr := errors.New("cache miss")
		updateErr := errors.New("some error")
		authMsg := "token request failed with error some error"

		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			nil,
			fetchErr,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(freshToken), nil)
		test.MockDynamoDBAPI.EXPECT().PutItem(gomock.Any()).Return(
			nil,
			updateErr,
		)

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.Error(t, err)
		require.EqualError(t, err, authMsg)
		require.Empty(t, res)
	})

	t.Run("Success on fetch from L2 cache (stale), fail on fetch from KMS", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		myErr := errors.New("some error")

		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(-2 * staleTimeout).Format(RFC3339_NANO_FORMAT))},
		}
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)

		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return(nil, myErr)

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		require.Equal(t, res, []byte(token))
	})

	t.Run("Success on fetch from L2 cache (Stale), Success on fetch from KMS, Success on update L2", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(-2 * staleTimeout).Format(RFC3339_NANO_FORMAT))},
		}
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(freshToken), nil)
		test.MockDynamoDBAPI.EXPECT().PutItem(gomock.Any()).Return(
			&dynamodb.PutItemOutput{},
			nil,
		)

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		require.Equal(t, res, []byte(freshToken))
	})

	t.Run("Failure on fetch from L2 cache (Expired), Success on fetch from KMS, Success on update L2", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(-20 * staleTimeout).Format(RFC3339_NANO_FORMAT))},
		}
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(freshToken), nil)
		test.MockDynamoDBAPI.EXPECT().PutItem(gomock.Any()).Return(
			&dynamodb.PutItemOutput{},
			nil,
		)

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.NoError(t, err)
		require.Equal(t, res, []byte(freshToken))
	})

	t.Run("Failure on fetch from L2 cache, Success on fetch from KMS, fail on update L2 (conditional), fail fetch L2", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		fetchErr := errors.New("cache miss")
		updateErr := awserr.New("ConditionalCheckFailedException", "The conditional request failed", nil)
		requestErr := awserr.NewRequestFailure(updateErr, 1, "testrequest")
		authMsg := "token request failed with error cache miss"

		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			nil,
			fetchErr,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return([]byte(freshToken), nil)
		test.MockDynamoDBAPI.EXPECT().PutItem(gomock.Any()).Return(
			nil,
			requestErr,
		)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			nil,
			fetchErr,
		)
		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.Error(t, err)
		require.EqualError(t, err, authMsg)
		require.Empty(t, res)
	})

	t.Run("Fail on fetch from L2 (expired), fail on update L2", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()
		expectedItem := map[string]*dynamodb.AttributeValue{
			"token":        {S: aws.String(token)},
			"refreshTime":  {N: aws.String(time.Now().Add(staleTimeout).Format(UNIX_SEC_TIME_FORMAT))},
			"creationTime": {S: aws.String(time.Now().Add(-20 * time.Hour).Format(RFC3339_NANO_FORMAT))},
		}
		test := newCachedL2AuthenticationsTest(ctrl, staleTimeout, expirationTimeout)
		test.MockDynamoDBAPI.EXPECT().GetItem(gomock.Any()).Return(
			&dynamodb.GetItemOutput{Item: expectedItem},
			nil,
		)
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), audienceHost).Return(nil, errors.New("some err"))

		res, err := test.CachedL2Authentications.Authenticate(ctx, audienceHost)
		require.EqualError(t, err, "token request failed with error some err")
		require.Empty(t, res)
	})

}

type cachedL2AuthenticationsTest struct {
	*CachedL2Authentications
	*authenticationmock.MockAuthenticationsAPI
	*dynamodbapimock.MockDynamoDBAPI
}

func newCachedL2AuthenticationsTest(
	ctrl *gomock.Controller,
	staleTimeout time.Duration,
	expirationTimeout time.Duration,
) *cachedL2AuthenticationsTest {
	mockAuthenticationsAPI := authenticationmock.NewMockAuthenticationsAPI(ctrl)
	mockDynamoDBAPI := dynamodbapimock.NewMockDynamoDBAPI(ctrl)
	return &cachedL2AuthenticationsTest{
		CachedL2Authentications: New(
			mockAuthenticationsAPI,
			staleTimeout,
			expirationTimeout,
			time.NewTicker(time.Millisecond),
			new(s2s2err.Logger),
			&operation.Starter{},
			mockDynamoDBAPI,
			"test-cache-table",
		),
		MockAuthenticationsAPI: mockAuthenticationsAPI,
		MockDynamoDBAPI:        mockDynamoDBAPI,
	}
}
