package s2s2dicaller

import (
	"fmt"
	"net/http"
	"net/url"
	"strings"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/authentication/authenticationiface"
)

// authenticatingHTTPClient provides an implementation of HTTPClient that
// authenticates all requests to a target
type authenticatingHTTPClient struct {
	Inner                  HTTPClient
	Authentications        authenticationiface.AuthenticationsAPI
	DangerouslyAllowNonTLS bool
}

func (h *authenticatingHTTPClient) Do(req *http.Request) (*http.Response, error) {
	if !h.DangerouslyAllowNonTLS && req.URL.Scheme != "https" {
		return nil, &ErrRefusingToSendToNonHTTPS{URL: *req.URL}
	}

	token, err := h.Authentications.Authenticate(req.Context(), canonicalizeWebOrigin(req.URL))
	if err != nil {
		return nil, err
	}

	req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", token))

	return h.Inner.Do(req)
}

var defaultPorts = map[string]string{
	"http":  "80",
	"https": "443",
}

func canonicalizeWebOrigin(url *url.URL) string {
	scheme, hostname, port := url.Scheme, url.Hostname(), url.Port()

	scheme = strings.ToLower(scheme)
	if defaultPorts[scheme] == port || port == "" {
		return strings.ToLower(fmt.Sprintf("%s://%s", scheme, hostname))
	}

	return strings.ToLower(fmt.Sprintf("%s://%s:%s", scheme, hostname, port))
}
