package s2s2dicaller

import (
	"errors"
	"net/http"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/mocks"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/mocks/authenticationmock"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAuthenticatingHTTPClientDo(t *testing.T) {
	const webOrigin = "https://weborigin"
	const token = "MYTOKEN"

	request := func() *http.Request {
		req, err := http.NewRequest(http.MethodGet, webOrigin, nil)
		require.NoError(t, err)
		return req
	}

	t.Run("success", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newAuthenticatingHTTPClientTest(ctrl)

		expectedRes := &http.Response{}
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), webOrigin).Return([]byte(token), nil)
		test.MockHTTPClient.EXPECT().
			Do(gomock.Any()).
			Do(func(r *http.Request) (*http.Response, error) {
				assert.Equal(t, "Bearer "+token, r.Header.Get("Authorization"))
				return nil, nil
			}).
			Return(expectedRes, nil)

		res, err := test.authenticatingHTTPClient.Do(request())
		require.NoError(t, err)
		assert.Equal(t, expectedRes, res)
	})

	t.Run("send to http only if configured", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newAuthenticatingHTTPClientTest(ctrl)

		expectedRes := &http.Response{}
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), "http://weborigin").Return([]byte(token), nil)
		test.MockHTTPClient.EXPECT().
			Do(gomock.Any()).
			Do(func(r *http.Request) (*http.Response, error) {
				assert.Equal(t, "Bearer "+token, r.Header.Get("Authorization"))
				return nil, nil
			}).
			Return(expectedRes, nil)

		request := request()
		request.URL.Scheme = "http"
		test.authenticatingHTTPClient.DangerouslyAllowNonTLS = true

		res, err := test.authenticatingHTTPClient.Do(request)
		require.NoError(t, err)
		assert.Equal(t, expectedRes, res)
	})

	t.Run("refuse to send to http", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newAuthenticatingHTTPClientTest(ctrl)

		request := request()
		request.URL.Scheme = "http"
		_, err := test.authenticatingHTTPClient.Do(request)
		assert.Equal(t, &ErrRefusingToSendToNonHTTPS{
			URL: *request.URL,
		}, err)
	})

	t.Run("authenticate failure", func(t *testing.T) {
		ctrl := gomock.NewController(t)
		defer ctrl.Finish()

		test := newAuthenticatingHTTPClientTest(ctrl)

		myErr := errors.New("myerr")
		test.MockAuthenticationsAPI.EXPECT().Authenticate(gomock.Any(), webOrigin).Return(nil, myErr)

		_, err := test.authenticatingHTTPClient.Do(request())
		assert.Equal(t, myErr, err)
	})
}

func TestCanonicalizeWebOrigin(t *testing.T) {
	parsedURL := func(in string) *url.URL {
		url, err := url.Parse(in)
		require.NoError(t, err)
		return url
	}

	tcs := []struct {
		URL      *url.URL
		Expected string
	}{
		{URL: parsedURL("https://my.host"), Expected: "https://my.host"},
		{URL: parsedURL("https://my.host/with/path"), Expected: "https://my.host"},
		{URL: parsedURL("HTTPS://MY.HOST"), Expected: "https://my.host"},
		{URL: parsedURL("https://my.host:443"), Expected: "https://my.host"},
		{URL: parsedURL("https://my.host:444"), Expected: "https://my.host:444"},
		{URL: parsedURL("http://my.host"), Expected: "http://my.host"},
		{URL: parsedURL("http://my.host:80"), Expected: "http://my.host"},
		{URL: parsedURL("http://my.host:81"), Expected: "http://my.host:81"},
	}

	for nTc, tc := range tcs {
		t.Run(strconv.Itoa(nTc), func(t *testing.T) {
			assert.Equal(t, tc.Expected, canonicalizeWebOrigin(tc.URL))
		})
	}
}

type authenticatingHTTPClientTest struct {
	*authenticatingHTTPClient
	*mocks.MockHTTPClient
	*authenticationmock.MockAuthenticationsAPI
}

func newAuthenticatingHTTPClientTest(ctrl *gomock.Controller) *authenticatingHTTPClientTest {
	mockHTTPClient := mocks.NewMockHTTPClient(ctrl)
	mockAuthenticationsAPI := authenticationmock.NewMockAuthenticationsAPI(ctrl)
	return &authenticatingHTTPClientTest{
		authenticatingHTTPClient: &authenticatingHTTPClient{
			Inner:           mockHTTPClient,
			Authentications: mockAuthenticationsAPI,
		},
		MockHTTPClient:         mockHTTPClient,
		MockAuthenticationsAPI: mockAuthenticationsAPI,
	}
}
