package s2s2dicaller

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/cachedauthentication/cachedauthenticationiface"
)

// Caller authenticates callers via s2s2 distributed identities
type Caller struct {
	authentications        cachedauthenticationiface.CachedAuthenticationsAPI
	dangerouslyAllowNonTLS bool
	// rate limits how often HardRefreshCache can be called
	hardRefreshCacheTicker *time.Ticker
}

// HTTPClient is the interface twirp expects
type HTTPClient interface {
	Do(*http.Request) (*http.Response, error)
}

// HTTPClient returns an HTTP client that handles authentication to a S2S2
// secured service.
func (c *Caller) HTTPClient(inner HTTPClient) HTTPClient {
	return &authenticatingHTTPClient{
		Authentications:        c.authentications,
		Inner:                  inner,
		DangerouslyAllowNonTLS: c.dangerouslyAllowNonTLS,
	}
}

// PrewarmForHost prewarms an authorization token for a single host.
func (c *Caller) PrewarmForHost(ctx context.Context, host string) error {
	_, err := c.authentications.Authenticate(ctx, host)
	return err
}

// HardRefreshCache will hard refresh all tokens in our cache. Running this in a
// goroutine every 10 minutes will optimize P100 latencies for your service
// since tokens won't need to be refetched in hot request paths.
func (c *Caller) HardRefreshCache(ctx context.Context) error {
	select {
	case <-ctx.Done():
		return ctx.Err()
	case <-c.hardRefreshCacheTicker.C:
	}
	return c.authentications.HardRefreshCache(ctx)
}
