package s2s2dicaller

import (
	"errors"
	"fmt"
	"time"

	logging "code.justin.tv/amzn/TwitchLogging"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/authentication"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/cachedauthentication"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/cachedl2authentication"
	"code.justin.tv/amzn/TwitchS2S2DistributedIdentitiesCaller/internal/s2s2err"
	"code.justin.tv/video/metrics-middleware/v2/awsmetric"
	"code.justin.tv/video/metrics-middleware/v2/operation"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/aws/aws-sdk-go/service/kms/kmsiface"
)

type identityOrigin string
type serviceDomain string
type serviceName string
type stage string
type cacheTableName string

const (
	staleTimeout      = time.Hour
	expirationTimeout = 24 * time.Hour
	// time period we will retry failed signatures at
	errorRateLimit = time.Minute
	// limit we will allow hard refreshes of caches at
	hardRefreshRate       = 10 * time.Minute
	defaultIdentityOrigin = identityOrigin("https://prod.s2s2identities.twitch.a2z.com")
	defaultServiceDomain  = serviceDomain("twitch")
)

var (
	errServiceNameMissing = errors.New("ServiceName is required for s2s2dicaller.Options")
	errStageMissing       = errors.New("Stage is required for s2s2dicaller.Options")
)

// Options for configuring Caller
type Options struct {
	// Required Configuration
	// This should be set to your service's name
	ServiceName string

	// Required Configuration
	// This should be set to the stage, i.e. beta or prod
	Stage string

	// Recommended Configuration
	// Operation starter for sending client metrics to your metrics platform.
	OperationStarter *operation.Starter

	// Recommended Configuration
	// Logger for sending errors that occur but don't cause an error in the
	// request path.
	Logger logging.Logger

	// Optional Configuration
	// Dangerously set this to true if you want to allow tokens to be sent to
	// non-https endpoints. DANGEROUS!!!! ONLY FOR LOCAL TESTING!!!!
	DangerouslyAllowNonTLS bool

	// ALWAYS LEAVE EMPTY.
	// The domain of all services.
	// Defaults to `twitch`.
	ServiceDomain string

	// ALWAYS LEAVE EMPTY.
	// Origin to pull identities from.
	// Defaults to `https://prod.s2s2identities.twitch.a2z.com`.
	IdentityOrigin string
}

func newCaller(
	options *Options,
	cachedAuthentications *cachedauthentication.CachedAuthentications,
	hardRefreshCacheTicker *time.Ticker,
) *Caller {
	return &Caller{
		authentications:        cachedAuthentications,
		dangerouslyAllowNonTLS: options.DangerouslyAllowNonTLS,
		hardRefreshCacheTicker: hardRefreshCacheTicker,
	}
}

func newAuthentications(
	serviceDomain serviceDomain,
	serviceName serviceName,
	stage stage,
	identityOrigin identityOrigin,
	kms kmsiface.KMSAPI,
) *authentication.Authentications {
	return &authentication.Authentications{
		ServiceDomain:  string(serviceDomain),
		ServiceName:    string(serviceName),
		Stage:          string(stage),
		IdentityOrigin: string(identityOrigin),
		KMS:            kms,
		Expiration:     expirationTimeout,
	}
}

func newCachedAuthentications(
	authentications *cachedl2authentication.CachedL2Authentications,
	logger *s2s2err.Logger,
	operationStarter *operation.Starter,
) *cachedauthentication.CachedAuthentications {
	return cachedauthentication.New(
		authentications,
		staleTimeout,
		expirationTimeout,
		time.NewTicker(errorRateLimit),
		logger,
		operationStarter,
	)
}

func newCachedL2Authentications(
	authentications *authentication.Authentications,
	logger *s2s2err.Logger,
	operationStarter *operation.Starter,
	dynamodbClient dynamodbiface.DynamoDBAPI,
	cacheTableName cacheTableName,
) *cachedl2authentication.CachedL2Authentications {
	return cachedl2authentication.New(
		authentications,
		staleTimeout,
		expirationTimeout,
		time.NewTicker(errorRateLimit),
		logger,
		operationStarter,
		dynamodbClient,
		string(cacheTableName),
	)
}

func newAwsMetricClient(operationStarter *operation.Starter) *awsmetric.Client {
	return &awsmetric.Client{
		Starter: operationStarter,
	}
}

func newKMS(
	session *session.Session,
	client *awsmetric.Client,
) kmsiface.KMSAPI {
	sess := client.AddToSession(session)
	return kms.New(sess)
}

func newOperationStarter(
	options *Options,
) *operation.Starter {
	if options.OperationStarter == nil {
		return &operation.Starter{}
	}

	return options.OperationStarter
}

type noopLogger struct{}

func (noopLogger) Log(string, ...interface{}) {}

func newLogger(
	options *Options,
) *s2s2err.Logger {
	return &s2s2err.Logger{Logger: options.Logger}
}

func newServiceDomain(options *Options) serviceDomain {
	if options.ServiceDomain != "" {
		return serviceDomain(options.ServiceDomain)
	}
	return defaultServiceDomain
}

func newIdentityOrigin(options *Options) identityOrigin {
	if options.IdentityOrigin != "" {
		return identityOrigin(options.IdentityOrigin)
	}
	return defaultIdentityOrigin
}

func newServiceName(options *Options) (serviceName, error) {
	if options.ServiceName == "" {
		return "", errServiceNameMissing
	}
	return serviceName(options.ServiceName), nil
}

func newStage(options *Options) (stage, error) {
	if options.Stage == "" {
		return "", errStageMissing
	}
	return stage(options.Stage), nil
}

func newHardRefreshCacheTicker() *time.Ticker {
	return time.NewTicker(hardRefreshRate)
}

func newDynamodb(session *session.Session, client *awsmetric.Client) dynamodbiface.DynamoDBAPI {
	sess := client.AddToSession(session)
	return dynamodb.New(sess)
}

func newCacheTableName(options *Options) (cacheTableName, error) {
	if options.ServiceName == "" {
		return "", errServiceNameMissing
	}
	if options.Stage == "" {
		return "", errStageMissing
	}
	table := fmt.Sprintf("S2S2DI-%s-%s-cache-table", options.ServiceName, options.Stage)
	return cacheTableName(table), nil

}
