package es256

import (
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"encoding/json"
	"errors"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestPrivateKey(t *testing.T) {
	t.Run("Sign and Validate value", func(t *testing.T) {
		generatedPrivateKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
		require.NoError(t, err)

		testEpk := &PrivateKey{
			PrivateKey: generatedPrivateKey,
		}

		testInput := []byte("awesome dog meme")

		signedInput, err := testEpk.Sign(testInput)
		require.NoError(t, err)

		err = testEpk.Validate(testInput, signedInput)
		require.NoError(t, err)
	})

	t.Run("Sign and Validate with different public key should fail", func(t *testing.T) {
		initialKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
		require.NoError(t, err)

		badKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
		require.NoError(t, err)

		initialEcdsaKey := &PrivateKey{
			PrivateKey: initialKey,
		}

		badEcdsaKey := &PrivateKey{
			PrivateKey: badKey,
		}

		testInput := []byte("awesome dog meme")
		signedInput, err := initialEcdsaKey.Sign(testInput)
		require.NoError(t, err)

		err = badEcdsaKey.Validate(testInput, signedInput)
		assert.Equal(t, ErrInvalidSig, err)
	})

	t.Run("Should marshal and unmarshal json", func(t *testing.T) {
		generatedPrivateKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
		require.NoError(t, err)

		initEPK := &PrivateKey{
			PrivateKey: generatedPrivateKey,
		}

		outEPK := &PrivateKey{}

		marshaledJSON, err := json.Marshal(initEPK)
		require.NoError(t, err)

		err = json.Unmarshal(marshaledJSON, outEPK)
		require.NoError(t, err)
	})

	t.Run("Should reconstruct key from stored values, and verify signatures correctly", func(t *testing.T) {
		generatedPrivateKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
		require.NoError(t, err)

		initEPK := &PrivateKey{
			PrivateKey: generatedPrivateKey,
		}

		testInput := []byte("awesome dog meme")

		signedInput, err := initEPK.Sign(testInput)
		require.NoError(t, err)

		marshaledKey, err := json.Marshal(initEPK)
		require.NoError(t, err)

		reconstructedEPK := &PrivateKey{}
		err = json.Unmarshal(marshaledKey, reconstructedEPK)
		require.NoError(t, err)

		err = reconstructedEPK.Validate(testInput, signedInput)
		require.NoError(t, err)

		secondarySignedInput, err := reconstructedEPK.Sign(testInput)
		require.NoError(t, err)

		err = initEPK.Validate(testInput, secondarySignedInput)
		require.NoError(t, err)
	})

	t.Run("json error", func(t *testing.T) {
		tcs := []struct {
			Name  string
			Value interface{}
			Err   error
		}{
			{
				Name: "incorrect kty",
				Value: struct {
					KeyType   string    `json:"kty"`
					Use       string    `json:"use"`
					Curve     string    `json:"crv"`
					KeyOps    stringSet `json:"key_ops"`
					Algorithm string    `json:"alg"`
				}{KeyType: "x"},
				Err: errors.New("Incorrect KeyType"),
			},
			{
				Name: "incorrect use",
				Value: struct {
					KeyType   string    `json:"kty"`
					Use       string    `json:"use"`
					Curve     string    `json:"crv"`
					KeyOps    stringSet `json:"key_ops"`
					Algorithm string    `json:"alg"`
				}{KeyType: "EC", Use: "x"},
				Err: errors.New("Incorrect Use"),
			},
			{
				Name: "incorrect crv",
				Value: struct {
					KeyType   string    `json:"kty"`
					Use       string    `json:"use"`
					Curve     string    `json:"crv"`
					KeyOps    stringSet `json:"key_ops"`
					Algorithm string    `json:"alg"`
				}{KeyType: "EC", Use: "sig", Curve: "x"},
				Err: errors.New("Incorrect Curve"),
			},
			{
				Name: "incorrect alg",
				Value: struct {
					KeyType   string    `json:"kty"`
					Use       string    `json:"use"`
					Curve     string    `json:"crv"`
					KeyOps    stringSet `json:"key_ops"`
					Algorithm string    `json:"alg"`
				}{KeyType: "EC", Use: "sig", Curve: "P-256", Algorithm: "x"},
				Err: errors.New("Incorrect Algorithm"),
			},
			{
				Name: "incorrect key_ops",
				Value: struct {
					KeyType   string    `json:"kty"`
					Use       string    `json:"use"`
					Curve     string    `json:"crv"`
					KeyOps    stringSet `json:"key_ops"`
					Algorithm string    `json:"alg"`
				}{KeyType: "EC", Use: "sig", Curve: "P-256", Algorithm: "ES256", KeyOps: *newStringSet("x")},
				Err: errors.New("Incorrect KeyOps"),
			},
		}

		for _, tc := range tcs {
			t.Run(tc.Name, func(t *testing.T) {
				marshaled, err := json.Marshal(tc.Value)
				require.NoError(t, err)

				var epk PrivateKey
				assert.Equal(t, tc.Err, json.Unmarshal(marshaled, &epk))
			})
		}
	})

	t.Run("Size", func(t *testing.T) {
		k := &PrivateKey{}
		assert.Equal(t, 64, k.Size())
	})

	t.Run("Name", func(t *testing.T) {
		k := &PrivateKey{}
		assert.Equal(t, "ES256", k.Name())
	})
}
