package TwitchTelemetryCloudWatchEMFSender

import (
	"bytes"
	"os"
	"testing"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

type mockLogger struct {
	mock.Mock
}

func (m *mockLogger) PutMetric(sample *telemetry.Sample) {
	m.Called(sample)
}
func (m *mockLogger) Flush() {
	m.Called()
}

func TestCloudWatchEMFSender_Flush(t *testing.T) {
	t.Run("Should flush the logger once", func(t *testing.T) {
		sender := New("test_service", new(bytes.Buffer))
		logger := new(mockLogger)
		logger.On("Flush").Return()
		sender.logger = logger
		sender.Flush()
		logger.AssertNumberOfCalls(t, "Flush", 1)
	})
}

func TestCloudWatchEMFSender_StopFlushes(t *testing.T) {
	t.Run("Should flush when stopped", func(t *testing.T) {
		sender := New("test_service", new(bytes.Buffer))
		logger := new(mockLogger)
		logger.On("Flush").Return()
		sender.logger = logger
		sender.Stop()
		logger.AssertNumberOfCalls(t, "Flush", 1)
	})
}

func TestCloudWatchEMFSender_ObserveSample(t *testing.T) {
	t.Run("should batch the metric on the logger if the sender is not stopped", func(t *testing.T) {
		sender := New("test_service", new(bytes.Buffer))
		logger := new(mockLogger)
		sample := &telemetry.Sample{}
		logger.On("PutMetric", sample).Return().Once()
		sender.logger = logger
		sender.ObserveSample(sample)
		logger.AssertExpectations(t)
	})

	t.Run("Should not batch the metric on the logger if the sender is stopped", func(t *testing.T) {
		sender := New("test_service", new(bytes.Buffer))
		logger := new(mockLogger)
		sample := &telemetry.Sample{}
		logger.On("PutMetric", sample).Return().Times(0)
		sender.logger = logger
		sender.isStopped()
		sender.ObserveSample(sample)
		logger.AssertExpectations(t)
	})
}

func TestCloudWatchEMFSender_Stop(t *testing.T) {
	t.Run("Should set the internal flag to 1", func(t *testing.T) {
		sender := New("test_service", os.Stdout)
		assert.Equal(t, uint32(0), sender.stopped)
		sender.Stop()
		assert.Equal(t, uint32(1), sender.stopped)
	})
}

func BenchmarkCloudWatchEMFSender_ObserveSample(b *testing.B) {
	b.ReportAllocs()
	sender := New("test_service", new(bytes.Buffer))
	for n := 0; n < b.N; n++ {
		sender.ObserveSample(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
	}
}

func BenchmarkCloudWatchEMFSender_Flush(b *testing.B) {
	b.ReportAllocs()
	sender := New("test_service", new(bytes.Buffer))
	for n := 0; n < b.N; n++ {
		b.StopTimer()
		sender.ObserveSample(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		sender.ObserveSample(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "latency",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		sender.ObserveSample(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "invocations",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		sender.ObserveSample(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "concurrentExecutions",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		b.StartTimer()
		sender.Flush()
	}
}
