package logger

import (
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"github.com/stretchr/testify/assert"
	"io/ioutil"
	"testing"
	"time"
)

func Test_bucket_putMetric(t *testing.T) {
	t.Run("Should add the sample to the internal store", func(t *testing.T) {
		testBucket := newBucket(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name: "duration",
			},
		}, time.Minute)
		testBucket.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name: "latency",
			},
		})
		assert.Equal(t, 1, len(testBucket.metrics["latency"]))
		testBucket.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name: "latency",
			},
		})
		assert.Equal(t, 2, len(testBucket.metrics["latency"]))
	})
}

func Test_bucket_serialize(t *testing.T) {
	successCase := func(b *bucket, want []byte) func(*testing.T) {
		return func(t *testing.T) {
			have := b.serialize("test_namespace")
			assert.JSONEqf(t, string(want), string(have), "actual and expected metric payload mismatch.\n expected %s\n, actual %s\n", want, have)
		}
	}

	testBucket := newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na"},
		},
		RollupDimensions: [][]string{{"region"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            100,
		Unit:             "Seconds",
	}, time.Minute)
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na"},
		},
		RollupDimensions: [][]string{{"region"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            200,
		Unit:             "Seconds",
	})
	want, _ := ioutil.ReadFile("testdata/one_metric_with_multiple_values.json")
	t.Run("One metric with multiple values", successCase(testBucket, want))

	testBucket = newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na"},
		},
		RollupDimensions: [][]string{{"region"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            100,
		Unit:             "Seconds",
	}, time.Minute)
	want, _ = ioutil.ReadFile("testdata/one_metric_with_one_value.json")
	t.Run("One metric with one value", successCase(testBucket, want))

	testBucket = newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na"},
		},
		RollupDimensions: [][]string{{"region"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            100,
		Unit:             "Seconds",
	}, time.Minute)
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production", "region": "na"},
		},
		RollupDimensions: [][]string{{"region"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            800,
		Unit:             "Milliseconds",
	})
	want, _ = ioutil.ReadFile("testdata/multiple_metrics_with_single_values.json")
	t.Run("Multiple metrics with single values", successCase(testBucket, want))

	testBucket = newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		RollupDimensions: [][]string{{"region"}, {"region", "app"}, {"app", "env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            100,
		Unit:             "Seconds",
	}, time.Minute)
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		RollupDimensions: [][]string{{"region"}, {"region", "app"}, {"app", "env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            800,
		Unit:             "Milliseconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		RollupDimensions: [][]string{{"region"}, {"region", "app"}, {"app", "env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            200,
		Unit:             "Seconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		RollupDimensions: [][]string{{"region"}, {"region", "app"}, {"app", "env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            250,
		Unit:             "Milliseconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		RollupDimensions: [][]string{{"region"}, {"region", "app"}, {"app", "env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            400,
		Unit:             "Milliseconds",
	})
	want, _ = ioutil.ReadFile("testdata/multiple_metrics_with_multiple_values.json")
	t.Run("Multiple metrics with multiple values", successCase(testBucket, want))

	testBucket = newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name: "latency",
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     100,
		Unit:      "Seconds",
	}, time.Minute)
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name: "duration",
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     800,
		Unit:      "Milliseconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name: "duration",
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     250,
		Unit:      "Milliseconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name: "duration",
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     400,
		Unit:      "Milliseconds",
	})
	want, _ = ioutil.ReadFile("testdata/multiple_metrics_without_dimensions_and_rollup.json")
	t.Run("Multiple metrics without dimensions and rollup dimensions", successCase(testBucket, want))

	testBucket = newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     100,
		Unit:      "Seconds",
	}, time.Minute)
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     200,
		Unit:      "Seconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     800,
		Unit:      "Milliseconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     250,
		Unit:      "Milliseconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		Timestamp: time.Unix(1593024684, 0),
		Value:     400,
		Unit:      "Milliseconds",
	})
	want, _ = ioutil.ReadFile("testdata/multiple_metrics_with_dimensions_and_without_rollup.json")
	t.Run("Multiple metrics with dimensions, but without rollup dimensions", successCase(testBucket, want))

	testBucket = newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production"},
		},
		RollupDimensions: [][]string{{"env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            100,
		Unit:             "Seconds",
	}, time.Minute)
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production"},
		},
		RollupDimensions: [][]string{{"env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            800,
		Unit:             "Milliseconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production"},
		},
		RollupDimensions: [][]string{{"env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            200,
		Unit:             "Seconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production"},
		},
		RollupDimensions: [][]string{{"env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            250,
		Unit:             "Milliseconds",
	})
	testBucket.putMetric(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production"},
		},
		RollupDimensions: [][]string{{"env"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            400,
		Unit:             "Milliseconds",
	})
	want, _ = ioutil.ReadFile("testdata/multiple_metrics_with_one_dimension_and_rollup.json")
	t.Run("Multiple metrics with one dimension, but without rollup dimensions", successCase(testBucket, want))
}

func Test_bucket_generateCloudWatchDimensionSetArray(t *testing.T) {
	testBucket := newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		RollupDimensions: [][]string{{"region"}, {"env"}, {"app"}, {"region", "app"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            100,
		Unit:             "Seconds",
	}, time.Minute)
	got := testBucket.generateCloudWatchDimensionSetArray()
	want := [][]string{{"app", "env", "region"}, {"app", "env"}, {"app", "region"}, {"env", "region"}, {"env"}}
	assert.Equal(t, want, got)
}

func Test_bucket_generateCloudWatchMetricDirectives(t *testing.T) {
	testBucket := newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "latency",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		RollupDimensions: [][]string{{"region"}, {"env"}, {"app"}, {"region", "app"}},
		Timestamp:        time.Unix(1593024684, 0),
		Value:            100,
		Unit:             "Seconds",
	}, time.Minute)
	got := testBucket.generateCloudWatchMetricDirectives("test_namespace")
	want := []metricDirective{{
		Namespace:  "test_namespace",
		Dimensions: [][]string{{"app", "env", "region"}, {"app", "env"}, {"app", "region"}, {"env", "region"}, {"env"}},
		Metrics: []metricDefinition{{
			Name: "latency",
			Unit: "Seconds",
		}},
	}}
	assert.Equal(t, want, got)
}

func Benchmark_bucket_putMetric(b *testing.B) {
	testBucket := newBucket(&telemetry.Sample{
		MetricID: telemetry.MetricID{
			Name:       "duration",
			Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
		},
		RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
		Timestamp:        time.Time{},
		Value:            float64(0),
		Unit:             "Seconds",
	}, time.Minute)
	for n := 0; n < b.N; n++ {
		testBucket.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
	}
}

func Benchmark_bucket_serialize(b *testing.B) {
	for n := 0; n < b.N; n++ {
		b.StopTimer()
		testBucket := newBucket(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(0),
			Unit:             "Seconds",
		}, time.Minute)
		testBucket.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		testBucket.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "latency",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		testBucket.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "invocations",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		testBucket.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "concurrentExecutions",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		b.StartTimer()
		testBucket.serialize("test_namespace")
	}
}
