package logger

import (
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"fmt"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"os"
	"testing"
	"time"
)

type mockWriter struct {
	mock.Mock
}

func (m *mockWriter) Write(bytes []byte) (n int, err error) {
	args := m.Called(bytes)
	return args.Int(0), args.Error(1)
}

func TestLogger_Flush(t *testing.T) {
	t.Run("Should write the serialized store to the sink", func(t *testing.T) {
		sink := new(mockWriter)
		logger := &Logger{
			store: newStore("test_namespace", time.Minute),
			sink:  sink,
		}
		logger.PutMetric(&telemetry.Sample{})
		logger.PutMetric(&telemetry.Sample{})
		storeCopy := copyStore(logger.store)
		logs := storeCopy.serialize()
		sink.On("Write", logs).Return(0, nil)
		logger.Flush()
		sink.AssertExpectations(t)
	})
}

func TestLogger_PutMetric(t *testing.T) {
	t.Run("Should truncate the dimensions if needed and prioritize the standard dimensions", func(t *testing.T) {
		sink := new(mockWriter)
		logger := &Logger{
			store: newStore("test_namespace", time.Minute),
			sink:  sink,
		}

		sample := &telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name: "latency",
				Dimensions: map[string]string{
					DimensionService:        "dimension_service",
					DimensionSubstage:       "dimension_substage",
					DimensionProcessAddress: "dimension_process_address",
					DimensionOperation:      "dimension_operation",
					DimensionRegion:         "dimension_region",
					DimensionDependency:     "dimension_dependency",
					DimensionStage:          "dimension_stage",
					"key8":                  "val8",
					"key9":                  "val9",
					"key10":                 "val10",
				},
			},
			RollupDimensions: nil,
			Timestamp:        time.Now(),
			Value:            100,
			Unit:             "Seconds",
		}

		standardDimensionsPresent := standardDimensions
		logger.PutMetric(sample)
		for _, dim := range standardDimensionsPresent {
			_, exists := sample.MetricID.Dimensions[dim]
			assert.Equal(t, true, exists)
		}
		assert.Equal(t, len(sample.MetricID.Dimensions), maxDimensions)
	})

	t.Run("Should flush the store if the bucket the metric belongs to is full and then add the metric", func(t *testing.T) {
		sink := new(mockWriter)
		sink.On("Write", mock.Anything).Return(0, nil)
		logger := &Logger{
			store: newStore("tes_namespace", time.Minute),
			sink:  sink,
		}

		for i := 0; i < maxMetricDefinitions+1; i++ {
			logger.PutMetric(&telemetry.Sample{
				MetricID: telemetry.MetricID{
					Name:       fmt.Sprintf("metric_number_%d", i),
					Dimensions: map[string]string{"key1": "val1"},
				},
			})
		}
		assert.Equal(t, len(logger.store.buckets), 1)
	})

	t.Run("Should flush the store if the metric the sample belongs to is full and then add the metric", func(t *testing.T) {
		sink := new(mockWriter)
		sink.On("Write", mock.Anything).Return(0, nil)
		logger := &Logger{
			store: newStore("tes_namespace", time.Minute),
			sink:  sink,
		}

		metricName := "test_metric"
		dimensions := map[string]string{"key1": "val1"}
		now := time.Now()
		for i := 0; i < maxMetricValues+1; i++ {
			logger.PutMetric(&telemetry.Sample{
				MetricID: telemetry.MetricID{
					Name:       metricName,
					Dimensions: dimensions,
				},
				Value:     100,
				Timestamp: now,
			})
		}
		got := len(logger.store.buckets[generateBucketKey(dimensions, nil, now)].metrics[metricName])
		assert.Equal(t, 1, got)
	})
}

func BenchmarkLogger_Flush(b *testing.B) {
	os.Stdout, _ = os.Open(os.DevNull)
	defer os.Stdout.Close()
	logger := New(os.Stdout, time.Minute, "test_namespace")
	for n := 0; n < b.N; n++ {
		b.StopTimer()
		logger.PutMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		logger.PutMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "latency",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		logger.PutMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "invocations",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		logger.PutMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "concurrentExecutions",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		b.StartTimer()
		logger.Flush()
	}
}

func BenchmarkLogger_PutMetric(b *testing.B) {
	logger := New(os.Stdout, time.Minute, "test_namespace")
	for n := 0; n < b.N; n++ {
		logger.PutMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
	}
}
