package logger

import (
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"sort"
	"strconv"
	"strings"
	"sync"
	"time"
)

// store stores all the batched metrics
type store struct {
	namespace  string
	buckets    buckets
	bucketKeys []string
	resolution time.Duration
	mux        sync.Mutex
}

// newStore instantiates a new store
func newStore(namespace string, resolution time.Duration) *store {
	return &store{
		namespace:  namespace,
		buckets:    make(buckets),
		resolution: resolution,
	}
}

// putMetric inserts the given sample to its corresponding bucket
func (s *store) putMetric(sample *telemetry.Sample) error {
	key := generateBucketKey(sample.MetricID.Dimensions, sample.RollupDimensions, sample.Timestamp)
	s.mux.Lock()
	defer s.mux.Unlock()
	bucket, bucketExists := s.buckets[key]
	if bucketExists {
		if bucket.numberOfUniqueMetrics() == maxMetricDefinitions {
			return maxMetricDefinitionsError{}
		} else if bucket.numberOfMetricValues(sample.MetricID.Name) == maxMetricValues {
			return maxMetricValuesError{}
		}
		bucket.putMetric(sample)
	} else {
		s.buckets[key] = newBucket(sample, s.resolution)
		s.bucketKeys = append(s.bucketKeys, key)
	}
	return nil
}

// serialize serializes all buckets into emf logs and returns a newline delimited buffer containing the logs
func (s *store) serialize() []byte {
	s.mux.Lock()
	defer s.mux.Unlock()
	var emfLogs []byte
	for _, bucketKey := range s.bucketKeys {
		bucket := s.buckets[bucketKey]
		emfLogs = append(emfLogs, bucket.serialize(s.namespace)...)
		emfLogs = append(emfLogs, byte('\n'))
	}
	s.buckets = make(buckets)
	s.bucketKeys = make([]string, 0)
	return emfLogs
}

// generateBucketKey generates a bucket key that determines the bucket a metric datum will be stored in.
// The bucket key is generated by concatenating the following segments: <dimensionPairs><rollup><timestamp>
func generateBucketKey(dimensions map[string]string, rollupDimensions [][]string, timestamp time.Time) string {
	var key strings.Builder

	// generate dimension pairs segment
	var dimensionKeys []string
	for key, _ := range dimensions {
		dimensionKeys = append(dimensionKeys, key)
	}
	sort.Strings(dimensionKeys)
	for _, dimension := range dimensionKeys {
		key.WriteString(dimension)
		key.WriteByte(0)
		key.WriteString(dimensions[dimension])
		key.WriteByte(0)
	}

	// generate rollup segment
	var rollupSegment []string
	for _, rollup := range rollupDimensions {
		sort.Strings(rollup)
		joinedRollup := strings.Join(rollup, "\000") + "\000"
		rollupSegment = append(rollupSegment, joinedRollup)
	}
	sort.Strings(rollupSegment)
	joinedRollupSegment := strings.Join(rollupSegment, "\000") + "\000"
	key.WriteString(joinedRollupSegment)

	// generate timestamp segment
	key.WriteString(strconv.FormatInt(timestamp.Unix(), 10))

	return key.String()
}
