package logger

import (
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"fmt"
	"github.com/stretchr/testify/assert"
	"io/ioutil"
	"testing"
	"time"
)

func Test_store_putMetric(t *testing.T) {
	t.Run("Should return an error when trying to insert to a full bucket", func(t *testing.T) {
		store := newStore("test_namespace", time.Minute)
		for i := 0; i < maxMetricDefinitions; i++ {
			err := store.putMetric(&telemetry.Sample{
				MetricID: telemetry.MetricID{
					Name:       fmt.Sprintf("metric_number_%d", i),
					Dimensions: map[string]string{"key1": "val1"},
				},
			})
			assert.Nil(t, err)
		}
		err := store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       fmt.Sprintf("metric_number_%d", maxMetricDefinitions+1),
				Dimensions: map[string]string{"key1": "val1"},
			},
		})
		assert.Equal(t, err, maxMetricDefinitionsError{})
	})

	t.Run("Should return an error when trying to insert to a key with more than <maxMetricValues> samples", func(t *testing.T) {
		store := newStore("test_namespace", time.Minute)
		for i := 0; i < maxMetricValues; i++ {
			err := store.putMetric(&telemetry.Sample{
				MetricID: telemetry.MetricID{
					Name:       "test_metric",
					Dimensions: map[string]string{"key1": "val1"},
				},
			})
			assert.Nil(t, err)
		}
		err := store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "test_metric",
				Dimensions: map[string]string{"key1": "val1"},
			},
		})
		assert.Equal(t, err, maxMetricValuesError{})
	})

	t.Run("Metrics with different dimensions (same rollup and timestamp) should be stored in different buckets", func(t *testing.T) {
		store := newStore("test_namespace", time.Minute)
		now := time.Now()
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1"},
			},
			Timestamp: now,
		})
		assert.Equal(t, 1, len(store.buckets))
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2"},
			},
			Timestamp: now,
		})
		assert.Equal(t, 2, len(store.buckets))
	})

	t.Run("Metrics with different rollup (same dimensions and timestamp) should be stored in different buckets", func(t *testing.T) {
		store := newStore("test_namespace", time.Minute)
		now := time.Now()
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2", "key3": "key3"},
			},
			RollupDimensions: [][]string{{"key1"}, {"key1", "key2"}},
			Timestamp:        now,
		})
		assert.Equal(t, 1, len(store.buckets))
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2", "key3": "key3"},
			},
			RollupDimensions: [][]string{{"key2"}, {"key2", "key3"}},
			Timestamp:        now,
		})
		assert.Equal(t, 2, len(store.buckets))
	})

	t.Run("Metrics with different timestamp (same dimensions and rollup) should be stored in different buckets", func(t *testing.T) {
		store := newStore("test_namespace", time.Minute)
		now := time.Now()
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2"},
			},
			RollupDimensions: [][]string{{"key1"}},
			Timestamp:        now,
		})
		assert.Equal(t, len(store.buckets), 1)
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2"},
			},
			RollupDimensions: [][]string{{"key1"}},
			Timestamp:        now.Add(time.Minute * 2),
		})
		assert.Equal(t, 2, len(store.buckets))
	})

	t.Run("metrics with the same timestamp, dimensions, and rollup should be stored in the same bucket", func(t *testing.T) {
		store := newStore("test_namespace", time.Minute)
		now := time.Now()
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2"},
			},
			RollupDimensions: [][]string{{"key1"}},
			Timestamp:        now,
		})
		assert.Equal(t, len(store.buckets), 1)
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2"},
			},
			RollupDimensions: [][]string{{"key1"}},
			Timestamp:        now,
		})
		assert.Equal(t, 1, len(store.buckets))
	})
}

func Test_store_serialize(t *testing.T) {
	t.Run("Should return a new line delimited buffer containing the emf logs", func(t *testing.T) {
		store := newStore("test_namespace", time.Minute)
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2"},
			},
			RollupDimensions: [][]string{{"key1"}},
			Timestamp:        time.Unix(1592948040, 0),
		})
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "MetricA",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2"},
			},
			Timestamp: time.Unix(1592948040, 0),
		})
		got := store.serialize()
		want, _ := ioutil.ReadFile("./testdata/test_store_serialize.txt")
		assert.Equal(t, want, got)
	})

	t.Run("Should clear all buckets", func(t *testing.T) {
		store := newStore("test_namespace", time.Minute)
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "latency",
				Dimensions: map[string]string{"key1": "val1", "key2": "val2"},
			},
			RollupDimensions: [][]string{{"key1"}},
			Timestamp:        time.Now(),
		})
		assert.Equal(t, 1, len(store.buckets))
		store.serialize()
		assert.Equal(t, 0, len(store.buckets))
	})
}

func Test_generateBucketKey(t *testing.T) {
	dimensions := map[string]string{"env": "production", "stage": "beta", "app": "test_app"}
	rollupDimensions := [][]string{{"env"}, {"stage"}, {"app"}, {"env", "app"}}
	timestamp := time.Unix(1593024660, 0)
	got := generateBucketKey(dimensions, rollupDimensions, timestamp)
	want := "app\000test_app\000env\000production\000stage\000beta\000app\000\000app\000env\000\000env\000\000stage\000\0001593024660"
	assert.Equal(t, want, got)
}

func Benchmark_store_serialize(b *testing.B) {
	store := newStore("test_namespace", time.Minute)
	for n := 0; n < b.N; n++ {
		b.StopTimer()
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "latency",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "invocations",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "concurrentExecutions",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
		b.StartTimer()
		store.serialize()
	}
}

func Benchmark_store_putMetric(b *testing.B) {
	store := newStore("test_namespace", time.Minute)
	for n := 0; n < b.N; n++ {
		store.putMetric(&telemetry.Sample{
			MetricID: telemetry.MetricID{
				Name:       "duration",
				Dimensions: map[string]string{"env": "production", "region": "na", "app": "test_app"},
			},
			RollupDimensions: [][]string{{"env"}, {"region"}, {"env, region"}},
			Timestamp:        time.Time{},
			Value:            float64(n),
			Unit:             "Seconds",
		})
	}
}

func Benchmark_generateBucketKey(b *testing.B) {
	dimensions := map[string]string{"env": "production", "stage": "beta", "app": "test_app", "dependency": "test_dependency"}
	rollupDimensions := [][]string{{"env"}, {"stage"}, {"app"}, {"env", "app"}}
	timestamp := time.Unix(1593024660, 0)
	for n := 0; n < b.N; n++ {
		generateBucketKey(dimensions, rollupDimensions, timestamp)
	}
}
