package middleware_test

import (
	"context"
	"fmt"
	_ "testing"
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	middleware "code.justin.tv/amzn/TwitchTelemetryMetricsMiddleware"
	"code.justin.tv/video/metrics-middleware/v2/operation"
)

// This example uses the middleware with no supporting hooks to show all of the parts needed to make it work.
// This may be useful when incorporating it into an existing service and/or when creating hooks for server or client
// libraries where there is no current support.  Ideally, the operation name and start/status/end is handled by
// hooks in the server and clients instead of being embedded in the code like this example.
func Example_minimal_usage_of_middleware() {
	processIdentifier := identifier.ProcessIdentifier{
		Service: "ExampleService",
		Stage:   "Stage",
		Region:  "Region",
	}

	// This is where the metrics are sent.
	// For MWS, import "code.justin.tv/amzn/TwitchTelemetryMWSMetricsSender" and use mws.New(&tPid, decoratingLogger)
	observer := &sampleLogger{}

	sampleReporter := telemetry.SampleReporter{
		SampleBuilder:  telemetry.SampleBuilder{ProcessIdentifier: processIdentifier},
		SampleObserver: observer,
	}

	metricsOpMonitor := &middleware.OperationMonitor{SampleReporter: sampleReporter}
	opStarter := &operation.Starter{OpMonitors: []operation.OpMonitor{metricsOpMonitor}}

	service := &exampleService{opStarter, processIdentifier, sampleReporter}
	fmt.Println(service.handleRequest("RequestPayload"))

	// Unordered output:
	// DependencyDuration: 1 Seconds
	// DependencySuccess: 1 Count
	// DependencyClientError: 0 Count
	// DependencyServerError: 0 Count
	// CustomMetric: 1234 Count
	// Duration: 1 Seconds
	// Success: 1 Count
	// ClientError: 0 Count
	// ServerError: 0 Count
	// ResponsePayload
}

// exampleService is a fake service that handles requests by calling a fake dependency.
type exampleService struct {
	opStarter         *operation.Starter
	processIdentifier identifier.ProcessIdentifier
	sampleReporter    telemetry.SampleReporter
}

func (service *exampleService) handleRequest(requestPayload string) string {
	// The operation name for the request being handled.
	operationName := "MyOperation"

	// Tell the middleware about the server operation.
	serverOpName := operation.Name{Kind: operation.KindServer, Group: service.processIdentifier.Service, Method: operationName}
	ctx, serverOp := service.opStarter.StartOp(context.Background(), serverOpName)
	defer serverOp.End()

	// Business logic here to handle the request.
	result := service.callDependency(ctx, requestPayload)

	// Report a custom metric.
	reporter := telemetry.SampleReporterWithContext(service.sampleReporter, ctx)
	reporter.Report("CustomMetric", 1234, telemetry.UnitCount)

	// Status uses gRPC codes.  Zero is success, so this has no effect and is just here as a placeholder for error checking.
	serverOp.SetStatus(operation.Status{Code: 0})
	return result
}

func (service *exampleService) callDependency(ctx context.Context, requestPayload string) string {
	// Names for the fake dependency and operation being invoked.
	dependency := "DependencyService"
	operationName := "DependencyOperation"

	// Tell the middleware about the client operation.
	clientOp := operation.Name{Kind: operation.KindClient, Group: dependency, Method: operationName}
	_, clientSpan := service.opStarter.StartOp(ctx, clientOp)
	defer clientSpan.End()

	// Call the fake dependency here to get the result.
	time.Sleep(500 * time.Millisecond)
	result := "ResponsePayload"

	// Status uses gRPC codes.  Zero is success, so this has no effect and is just here as a placeholder for error checking.
	clientSpan.SetStatus(operation.Status{Code: 0})
	return result
}

// sampleLogger prints a line for each sample it sees.
type sampleLogger struct{}

func (observer *sampleLogger) ObserveSample(sample *telemetry.Sample) {
	// The value is rounded to a whole number so that Duration should be consistently reported.
	fmt.Printf("%v: %.0f %v\n", sample.MetricID.Name, sample.Value, sample.Unit)
}

func (observer *sampleLogger) Flush() {}
func (observer *sampleLogger) Stop()  {}
