package poller

import (
	"fmt"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"
)

func TestJitterTickerMinEqualDelay(t *testing.T) {
	Convey("Given a JitterTicker with min equal to delay", t, func() {
		start := time.Now()
		delay := 100 * time.Millisecond
		min := delay
		ticker := NewJitterTicker(delay, min)

		Convey("Delays should be approximately the delay", func() {
			defer ticker.Stop()
			for i := 0; i < 10; i++ {
				select {
				case t := <-ticker.C:
					So(t.Sub(start), ShouldBeBetween, min*9/10 /* tolerance */, delay*11/10 /* tolerance */)
					start = time.Now()
				}
			}
		})
	})
}

func TestJitterTickerMinGreaterThanDelay(t *testing.T) {
	Convey("Given a JitterTicker with min larger than delay", t, func() {
		start := time.Now()
		delay := 100 * time.Millisecond
		min := delay * 2
		ticker := NewJitterTicker(delay, min)

		Convey("Delays should be approximately the delay", func() {
			defer ticker.Stop()
			for i := 0; i < 10; i++ {
				select {
				case t := <-ticker.C:
					So(t.Sub(start), ShouldBeBetween, delay*9/10 /* tolerance */, delay*11/10 /* tolerance */)
					start = time.Now()
				}
			}
		})
	})
}

func TestJitterTickerMinLessThanDelay(t *testing.T) {
	Convey("Given a JitterTicker with min less than delay", t, func() {
		start := time.Now()
		delay := 100 * time.Millisecond
		min := 90 * time.Millisecond
		ticker := NewJitterTicker(delay, min)

		Convey("Delays should be approximately between the min and the delay", func() {
			fmt.Println()
			defer ticker.Stop()
			for i := 0; i < 10; i++ {
				select {
				case t := <-ticker.C:
					So(t.Sub(start), ShouldBeBetween, min*9/10 /* tolerance */, delay*11/10 /* tolerance */)
					start = time.Now()
				}
			}
		})
	})
}

func TestJitterTickerDelayedStart(t *testing.T) {
	Convey("Given a JitterTicker with min equal to delay", t, func() {
		delay := 100 * time.Millisecond
		min := delay
		ticker := NewJitterTicker(delay, min)

		Convey("Wait for a second", func() {
			time.Sleep(time.Second)
			Convey("First timing after delayed start will trigger quickly", func() {
				start := time.Now()
				defer ticker.Stop()
				t := <-ticker.C
				So(t.Sub(start), ShouldBeLessThan, delay*11/10 /* tolerance */)
				Convey("Remaining delays should be accurate", func() {
					start := time.Now()
					for i := 0; i < 10; i++ {
						select {
						case t := <-ticker.C:
							So(t.Sub(start), ShouldBeBetween, min*9/10 /* tolerance */, delay*11/10 /* tolerance */)
							start = time.Now()
						}
					}
				})
			})
		})
	})
}
