package poller

import (
	"testing"
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"

	"sync"

	. "github.com/smartystreets/goconvey/convey"
)

// always returns 3 samples: Foobar, Garply, and Waldo, with respective values 1, 2, and 3,
// and respective units UnitCount, UnitCount, UnitSeconds
// Waldo has a single extra dimension, Wubbalubba=Dubdub
type dummyPollingJob struct {
	fetchCount int
}

func (dpj *dummyPollingJob) Fetch() ([]*telemetry.Sample, error) {
	dpj.fetchCount++
	waldoMetricID := telemetry.NewMetricID("Waldo")
	waldoMetricID.AddDimension("Wubbalubba", "Dubdub")
	return []*telemetry.Sample{
		&telemetry.Sample{
			MetricID: *telemetry.NewMetricID("Foobar"),
			Value:    float64(1),
			Unit:     telemetry.UnitCount,
		},
		&telemetry.Sample{
			MetricID: *telemetry.NewMetricID("Garply"),
			Value:    float64(2),
			Unit:     telemetry.UnitCount,
		},
		&telemetry.Sample{
			MetricID: *waldoMetricID,
			Value:    float64(3),
			Unit:     telemetry.UnitSeconds,
		},
	}, nil
}

func TestPollingCollectorFetchSamples(t *testing.T) {
	Convey("Given a PollingCollector using a dummy PollingJob", t, func() {

		dpj := &dummyPollingJob{}
		pc := NewPollingCollector(dpj, 3*time.Second, nil, nil, nil)

		Convey("With a sample builder", func() {

			builder := &telemetry.SampleBuilder{
				ProcessIdentifier: identifier.ProcessIdentifier{
					Service: "CoolService",
					Stage:   "production",
					Region:  "us-west-2",
				},
				OperationName: "CoolOperation",
			}
			pc.WithSampleBuilder(builder)

			Convey("When data is fetched for some timestamp", func() {

				t := time.Now().Add(-5 * time.Minute)
				samples, err := pc.fetchSamples(t)

				Convey("Then there should be no errors", func() {
					So(err, ShouldBeNil)
				})

				Convey("Then 3 samples should be returned", func() {
					So(len(samples), ShouldEqual, 3)
				})

				Convey("Then each sample should have the base dimensions", func() {

					So(samples[0].MetricID.Name, ShouldResemble, "Foobar")
					So(samples[1].MetricID.Name, ShouldResemble, "Garply")
					So(samples[2].MetricID.Name, ShouldResemble, "Waldo")

					for _, sample := range samples {
						So(sample.MetricID.Dimensions["Service"], ShouldResemble, "CoolService")
						So(sample.MetricID.Dimensions["Region"], ShouldResemble, "us-west-2")
						So(sample.MetricID.Dimensions["Stage"], ShouldResemble, "production")
						So(sample.MetricID.Dimensions["Operation"], ShouldResemble, "CoolOperation")
					}

					Convey("And the Waldo metric should have an extra dimension", func() {
						So(samples[2].MetricID.Dimensions["Wubbalubba"], ShouldResemble, "Dubdub")
					})

					Convey("And the timestamp should be set properly", func() {
						for _, sample := range samples {
							So(sample.Timestamp, ShouldResemble, t)
						}
					})
				})
			})

			Convey("When no data is attempted to be built", func() {
				t := time.Now().Add(-5 * time.Minute)
				samples, err := pc.buildFullSample(nil, t)

				Convey("Then there should be an errors and no samples", func() {
					So(err, ShouldNotBeNil)
					So(samples, ShouldBeNil)
				})
			})
		})
	})
}

type mockObserver struct {
	mux             sync.Mutex
	samplesObserved int
}

func (mo *mockObserver) ObserveSample(s *telemetry.Sample) {
	mo.mux.Lock()
	defer mo.mux.Unlock()
	mo.samplesObserved++
}

func (mo *mockObserver) GetSampleObserved() int {
	mo.mux.Lock()
	defer mo.mux.Unlock()
	numCp := mo.samplesObserved
	return numCp
}

func (mo *mockObserver) Flush() {}
func (mo *mockObserver) Stop()  {}

func TestPollingCollectorTiming(t *testing.T) {
	Convey("Given a PollingCollector with a dummy PollingJob and a 1.7 second interval", t, func() {

		dpj := &dummyPollingJob{}
		pc := NewPollingCollector(dpj, 1700*time.Millisecond, nil, nil, nil)

		Convey("And an observer that records sample counts", func() {

			obs := &mockObserver{}
			pc.WithSampleObserver(obs)
			pc.WithSampleBuilder(&telemetry.SampleBuilder{
				ProcessIdentifier: identifier.ProcessIdentifier{
					Service: "Foobar",
				},
			})

			Convey("When we don't start the PollingCollector", func() {

				// do nothing

				Convey("And wait 2 seconds", func() {

					time.Sleep(2000 * time.Millisecond)

					Convey("Then the observer should receive no samples", func() {
						So(obs.GetSampleObserved(), ShouldEqual, 0)
					})
				})
			})
			// Start a crazy timing sequence where we wait for time to elapse and let the
			// PollingCollector do it's thing
			// In theory this is kind of sketchy because we are relying on time.Sleep() to
			// behave nicely... but the time window is so large (3-5 seconds) that it should
			// not be a problem.
			Convey("When we start the PollingCollector and wait, samples should come in every interval", func() {

				// hacky way to wait until the interval is about to start before firing up the PollingCollector...
				now := time.Now()
				startTime := now.Truncate(1700 * time.Millisecond).Add(1600 * time.Millisecond)
				if now.Before(startTime) {
					time.Sleep(startTime.Sub(now))
				}
				pc.Start()

				time.Sleep(2000 * time.Millisecond)

				So(obs.GetSampleObserved(), ShouldEqual, 3)

				time.Sleep(2000 * time.Millisecond)

				So(obs.GetSampleObserved(), ShouldEqual, 6)

				pc.Stop()

				time.Sleep(2000 * time.Millisecond)

				So(obs.GetSampleObserved(), ShouldEqual, 9)
			})
		})
	})
}

func TestPollingCollectorTimingWithJitter(t *testing.T) {
	Convey("Given a PollingCollector with a dummy PollingJob and a 1.6-1.7 second interval", t, func() {

		minInterval := 1600 * time.Millisecond
		interval := 1700 * time.Millisecond
		pause := 2000 * time.Millisecond
		dpj := &dummyPollingJob{}
		pc := NewPollingCollector(dpj, interval, nil, nil, nil).WithMinInterval(minInterval)

		Convey("And an observer that records sample counts", func() {
			obs := &mockObserver{}
			pc.WithSampleObserver(obs)
			pc.WithSampleBuilder(&telemetry.SampleBuilder{
				ProcessIdentifier: identifier.ProcessIdentifier{
					Service: "Foobar",
				},
			})

			Convey("When we don't start the PollingCollector", func() {

				// do nothing

				Convey("And wait 2 seconds", func() {
					time.Sleep(2000 * time.Millisecond)

					Convey("Then the observer should receive no samples", func() {
						So(obs.GetSampleObserved(), ShouldEqual, 0)
					})
				})
			})
			// Start a crazy timing sequence where we wait for time to elapse and let the
			// PollingCollector do it's thing
			// In theory this is kind of sketchy because we are relying on time.Sleep() to
			// behave nicely... but the time window is so large (3-5 seconds) that it should
			// not be a problem.
			Convey("When we start the PollingCollector and wait, samples should come in every interval", func() {
				// hacky way to wait until the interval is about to start before firing up the PollingCollector...
				now := time.Now()
				startTime := now.Truncate(interval).Add(interval - 100*time.Millisecond)
				if now.Before(startTime) {
					time.Sleep(startTime.Sub(now))
				}
				pc.Start()

				time.Sleep(pause)
				So(obs.GetSampleObserved(), ShouldEqual, 3)

				time.Sleep(pause)
				So(obs.GetSampleObserved(), ShouldEqual, 6)

				pc.Stop()

				time.Sleep(pause)
				So(obs.GetSampleObserved(), ShouldEqual, 9)
			})
		})
	})
}

func TestPollingStartStop(t *testing.T) {
	Convey("Given a PollingCollector no sample builder or observer", t, func() {

		dpj := &dummyPollingJob{}
		pc := NewPollingCollector(dpj, 1700*time.Millisecond, nil, nil, nil)

		Convey("When start is called", func() {

			pc.Start()

			Convey("Then the polling collector should not start running", func() {
				So(pc.IsRunning(), ShouldEqual, false)
			})
		})

		Convey("When a sample builder is added", func() {
			pc.WithSampleBuilder(&telemetry.SampleBuilder{})

			Convey("Then Start still wont work", func() {
				pc.Start()
				So(pc.IsRunning(), ShouldEqual, false)
			})
		})

		Convey("When a sample observer is added", func() {
			pc.WithSampleObserver(&mockObserver{})

			Convey("Then Start still wont work", func() {
				pc.Start()
				So(pc.IsRunning(), ShouldEqual, false)
			})
		})

		Convey("When a sample observer and a sample builder are added", func() {
			pc.WithSampleObserver(&mockObserver{})
			pc.WithSampleBuilder(&telemetry.SampleBuilder{})

			Convey("Then Start will work", func() {
				pc.Start()
				So(pc.IsRunning(), ShouldEqual, true)
			})
		})
	})
}

func TestAlignTicker(t *testing.T) {
	Convey("Given a default PollingCollector", t, func() {
		pc := NewPollingCollector(&dummyPollingJob{}, 1700*time.Millisecond, nil, nil, nil)

		Convey("Before AlignTicker is called interval alignment must be true", func() {
			So(pc.alignTicker, ShouldEqual, true)
		})

		Convey("After AlignTicker is called with false alignment must be false", func() {
			pc.AlignTicker(false)
			So(pc.IsRunning(), ShouldEqual, false)
		})
	})
}
