package statsd

import (
	"sort"
	"testing"

	"code.justin.tv/amzn/TwitchTelemetry"
	. "github.com/smartystreets/goconvey/convey"
)

func TestDimensionNameSorting(t *testing.T) {
	Convey("Given an empty set of dimensions in a sample", t, func() {

		dimensions := make(telemetry.DimensionSet)
		distribution := &telemetry.Distribution{
			MetricID: telemetry.MetricID{
				Name:       "Foobar",
				Dimensions: telemetry.DimensionSet(dimensions),
			},
		}

		Convey("When using a metricBuilder", func() {

			mb := newMetricBuilder(distribution, nil)

			Convey("The metric structure should just be the ProcessIdentifier fields", func() {
				correctOrder := []string{
					telemetry.DimensionService,
					telemetry.DimensionStage,
					telemetry.DimensionSubstage,
					telemetry.DimensionRegion,
				}
				So(mb.nodeList, ShouldResemble, correctOrder)
			})
		})

	})

	Convey("Given a set of dimensions containing just ProcessIdentifier fields in a distribution", t, func() {

		dimensions := telemetry.DimensionSet{
			"Stage":    "prod",
			"Substage": "canary",
			"Service":  "foobar",
			"Region":   "us-west-2",
		}

		distribution := &telemetry.Distribution{
			MetricID: telemetry.MetricID{
				Name:       "Foobar",
				Dimensions: telemetry.DimensionSet(dimensions),
			},
		}

		Convey("When using a metricBuilder", func() {

			mb := newMetricBuilder(distribution, nil)

			Convey("The sorted dimension list should be in proper ProcessIdentifier order", func() {

				correctOrder := []string{
					telemetry.DimensionService,
					telemetry.DimensionStage,
					telemetry.DimensionSubstage,
					telemetry.DimensionRegion,
				}
				So(mb.nodeList, ShouldResemble, correctOrder)
			})
		})

	})

	Convey("Given a subset of ProcessIdentifier dimensions in a distribution", t, func() {

		dimensions := telemetry.DimensionSet{
			"Stage":   "prod",
			"Service": "foobar",
			"Region":  "us-west-2",
		}

		dimensions = make(telemetry.DimensionSet)
		distribution := &telemetry.Distribution{
			MetricID: telemetry.MetricID{
				Name:       "Foobar",
				Dimensions: dimensions,
			},
		}

		Convey("When passed to a metricBuilder", func() {

			mb := newMetricBuilder(distribution, nil)

			Convey("The provided dimensions are sorted with missing ProcessIdentifier fields added", func() {
				correctOrder := []string{
					telemetry.DimensionService,
					telemetry.DimensionStage,
					telemetry.DimensionSubstage,
					telemetry.DimensionRegion,
				}
				So(mb.nodeList, ShouldResemble, correctOrder)
			})
		})

	})

	Convey("Given only non-ProcessIdentifier dimensions in a distribution", t, func() {

		dimensions := telemetry.DimensionSet{
			"C": "c",
			"B": "b",
			"A": "a",
		}

		distribution := &telemetry.Distribution{
			MetricID: telemetry.MetricID{
				Name:       "Foobar",
				Dimensions: dimensions,
			},
		}

		Convey("When passed to a metricBuilder", func() {

			mb := newMetricBuilder(distribution, nil)

			Convey("The dimensions are sorted alphabetically, with ProcessIdentifier fields added", func() {
				correctOrder := []string{
					telemetry.DimensionService,
					telemetry.DimensionStage,
					telemetry.DimensionSubstage,
					telemetry.DimensionRegion,
					"A",
					"B",
					"C",
				}
				So(mb.nodeList, ShouldResemble, correctOrder)
			})

		})

	})

	Convey("Given a subset of ProcessIdentifier dimensions and some extra dimensions in a distribution", t, func() {

		dimensions := telemetry.DimensionSet{
			"Stage":    "prod",
			"Substage": "canary",
			"Service":  "foobar",
			"Region":   "us-west-2",
			"D":        "d",
			"C":        "c",
			"A":        "a",
			"B":        "b",
		}

		distribution := &telemetry.Distribution{
			MetricID: telemetry.MetricID{
				Name:       "Foobar",
				Dimensions: dimensions,
			},
		}

		Convey("When passed to a metricBuilder", func() {

			mb := newMetricBuilder(distribution, nil)

			Convey("Then the nodeList should start with sorted ProcessIdentifier dimensions", func() {
				So(
					mb.nodeList[0:4],
					ShouldResemble,
					[]string{
						telemetry.DimensionService,
						telemetry.DimensionStage,
						telemetry.DimensionSubstage,
						telemetry.DimensionRegion,
					},
				)
			})

			Convey("The nodeList should end with the alphabetically sorted extra dimensions", func() {
				So(mb.nodeList[4:8], ShouldResemble, []string{"A", "B", "C", "D"})
			})

		})
	})

}

func TestExpandMetricName(t *testing.T) {

	Convey("Given a distribution", t, func() {

		dimensions := make(telemetry.DimensionSet)
		distribution := &telemetry.Distribution{
			MetricID: telemetry.MetricID{
				Name:       "Foobar",
				Dimensions: dimensions,
			},
		}

		Convey("With a full ProcessIdentifier of dimensions", func() {
			dimensions := telemetry.DimensionSet{
				"Stage":    "prod",
				"Substage": "canary",
				"Service":  "foobar",
				"Region":   "us-west-2",
			}

			distribution.MetricID.Dimensions = dimensions

			Convey("With no rollups", func() {
				distribution.RollupDimensions = [][]string{}

				Convey("When passed to a metricBuilder and metric names are generated", func() {
					mb := newMetricBuilder(distribution, nil)
					metricNames := mb.expandMetricNames()

					Convey("The only metric is the fully qualified metric", func() {
						correctName := "Service/foobar.Stage/prod.Substage/canary.Region/us-west-2.Foobar"
						So(len(metricNames), ShouldEqual, 1)
						So(metricNames[0], ShouldEqual, correctName)
					})
				})
			})

			Convey("With rollups on [Region, Substage] and [Substage]", func() {
				distribution.RollupDimensions = [][]string{
					[]string{telemetry.DimensionRegion, telemetry.DimensionSubstage},
					[]string{telemetry.DimensionSubstage},
				}

				Convey("When passed to a metricBuilder and metric names are generated", func() {
					mb := newMetricBuilder(distribution, nil)
					metricNames := mb.expandMetricNames()
					sort.Strings(metricNames)

					correctMetricNames := []string{
						"Service/foobar.Stage/prod.Substage/canary.Region/us-west-2.Foobar",
						"Service/foobar.Stage/prod.Substage/all.Region/all.Foobar",
						"Service/foobar.Stage/prod.Substage/all.Region/us-west-2.Foobar",
					}
					sort.Strings(correctMetricNames)

					Convey("There are three generated metrics", func() {

						So(len(metricNames), ShouldEqual, 3)
						So(metricNames, ShouldResemble, correctMetricNames)
					})
				})
			})

		})

		Convey("With a partial set of ProcessIdentifier Dimensions", func() {

			distribution.MetricID.Dimensions = telemetry.DimensionSet{
				"Stage":   "prod",
				"Service": "foobar",
				"Region":  "us-west-2",
			}

			Convey("With no rollups", func() {
				distribution.RollupDimensions = [][]string{}

				Convey("When passed to a metricBuilder and metric names are generated", func() {
					mb := newMetricBuilder(distribution, nil)
					metricNames := mb.expandMetricNames()

					Convey("There should be one metric name where the omitted ProcessIdentifier fields are 'none'", func() {
						correctMetricName := "Service/foobar.Stage/prod.Substage/none.Region/us-west-2.Foobar"
						So(len(metricNames), ShouldEqual, 1)
						So(metricNames[0], ShouldEqual, correctMetricName)
					})
				})
			})

			Convey("With rollups on omitted fields", func() {
				distribution.RollupDimensions = [][]string{
					[]string{telemetry.DimensionSubstage},
				}

				Convey("When passed to a metricBuilder and metric names are generated", func() {
					mb := newMetricBuilder(distribution, nil)
					metricNames := mb.expandMetricNames()
					sort.Strings(metricNames)

					Convey("Then the omitted fields should also be rolled up in 'all' fields", func() {
						correctMetricNames := []string{
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.Foobar",
							"Service/foobar.Stage/prod.Substage/all.Region/us-west-2.Foobar",
						}
						sort.Strings(correctMetricNames)
						So(len(metricNames), ShouldEqual, 2)
						So(metricNames, ShouldResemble, correctMetricNames)
					})
				})
			})
		})

		Convey("With a partial ProcessIdentifier and extra dimensions", func() {
			distribution.MetricID.Dimensions = telemetry.DimensionSet{
				"Stage":   "prod",
				"Service": "foobar",
				"Region":  "us-west-2",
				"C":       "c",
				"B":       "b",
				"A":       "a",
			}

			Convey("With no rollups", func() {
				distribution.RollupDimensions = [][]string{}

				Convey("When passed to a metricBuilder and metric names are generated", func() {
					mb := newMetricBuilder(distribution, nil)
					metricNames := mb.expandMetricNames()

					Convey("Then there should be a single generated metric", func() {
						correctMetricNames := []string{
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/a.B/b.C/c.Foobar",
						}
						So(len(metricNames), ShouldEqual, 1)
						So(metricNames, ShouldResemble, correctMetricNames)
					})
				})
			})

			Convey("With many rollups", func() {
				distribution.RollupDimensions = [][]string{
					[]string{"A", "B"},
					[]string{"A", "B", "C"},
					[]string{"A", "C"},
					[]string{"B", "C"},
					[]string{"C"},
					[]string{"A"},
					[]string{"B"},
				}

				Convey("When passed to a metricBuilder and metric names are generated", func() {
					mb := newMetricBuilder(distribution, nil)
					metricNames := mb.expandMetricNames()
					sort.Strings(metricNames)

					Convey("Then there should be many generated metrics", func() {
						correctMetricNames := []string{
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/a.B/b.C/c.Foobar",
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/all.B/all.C/c.Foobar",
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/all.B/all.C/all.Foobar",
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/all.B/b.C/all.Foobar",
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/a.B/all.C/all.Foobar",
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/a.B/b.C/all.Foobar",
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/all.B/b.C/c.Foobar",
							"Service/foobar.Stage/prod.Substage/none.Region/us-west-2.A/a.B/all.C/c.Foobar",
						}
						sort.Strings(correctMetricNames)
						So(len(metricNames), ShouldEqual, len(correctMetricNames))
						So(metricNames, ShouldResemble, correctMetricNames)
					})
				})
			})
		})
	})

	Convey("When given a distribution with a dimension set containing some empty ProcessIdentifier fields", t, func() {
		dimensions := telemetry.DimensionSet{
			"Stage":    "prod",
			"Service":  "foobar",
			"Region":   "us-west-2",
			"Substage": "",
		}

		distribution := &telemetry.Distribution{
			MetricID: telemetry.MetricID{
				Name:       "Foobar",
				Dimensions: dimensions,
			},
		}

		Convey("When passed to a metricBuilder", func() {
			mb := newMetricBuilder(distribution, nil)

			Convey("And metric names are expanded", func() {
				metricNames := mb.expandMetricNames()

				Convey("Then the empty string ProcessIdentifier fields should be mapped to noneStrings", func() {
					correctName := "Service/foobar.Stage/prod.Substage/none.Region/us-west-2.Foobar"
					So(len(metricNames), ShouldEqual, 1)
					So(metricNames[0], ShouldResemble, correctName)
				})
			})
		})

		Convey("With a full ProcessIdentifier, plus Operation and ProcessAddress and extra dimensions", func() {
			dimensions := telemetry.DimensionSet{
				"Stage":          "prod",
				"Substage":       "canary",
				"Service":        "foobar",
				"Region":         "us-west-2",
				"ProcessAddress": "example_com",
				"Operation":      "GetFoobars",
				"A":              "a",
				"B":              "b",
			}

			distribution.MetricID.Dimensions = dimensions

			Convey("When passed to a metricBuilder and metric names are generated", func() {
				mb := newMetricBuilder(distribution, nil)
				metricNames := mb.expandMetricNames()

				Convey("Then there should be one metric", func() {
					So(len(metricNames), ShouldEqual, 1)
				})

				Convey("Then the ProcessAddress and Operation come between the ProcessIdentifier and extra dimensions", func() {
					correctMetricName := "Service/foobar.Stage/prod.Substage/canary.Region/us-west-2.ProcessAddress/example_com.Operation/GetFoobars.A/a.B/b.Foobar"
					So(metricNames[0], ShouldResemble, correctMetricName)
				})

			})
		})
		Convey("With dimensons containing periods", func() {
			dimensions := telemetry.DimensionSet{
				"Stage":          "prod.Prod",
				"Substage":       "canary.Canary",
				"Service":        "foobar",
				"Region":         "us.west.2",
				"Cool.Dimension": "Cool.Value",
			}

			distribution.MetricID.Dimensions = dimensions

			Convey("When passed to a metricBuilder and metric names are generated", func() {
				mb := newMetricBuilder(distribution, nil)
				metricNames := mb.expandMetricNames()

				Convey("Then  periods should be remapped to underscores", func() {
					correctMetricName := "Service/foobar.Stage/prod_Prod.Substage/canary_Canary.Region/us_west_2.Cool_Dimension/Cool_Value.Foobar"
					So(metricNames[0], ShouldResemble, correctMetricName)
				})

			})
		})
	})

	Convey("Given a distribution for a metric with a period in the name", t, func() {

		dimensions := make(telemetry.DimensionSet)
		distribution := &telemetry.Distribution{
			MetricID: telemetry.MetricID{
				Name:       "Foobar.Awesome",
				Dimensions: dimensions,
			},
		}

		Convey("When passed to a metricBuilder and metric names are generated", func() {
			mb := newMetricBuilder(distribution, nil)
			metricNames := mb.expandMetricNames()

			Convey("Then the periods should be mapped to underscores", func() {
				correctMetricName := "Service/none.Stage/none.Substage/none.Region/none.Foobar_Awesome"
				So(metricNames[0], ShouldResemble, correctMetricName)
			})
		})
	})
}
