package agent

import (
	"testing"

	"github.com/shirou/gopsutil/cpu"
	. "github.com/smartystreets/goconvey/convey"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
)

func TestBaseCPUSample(t *testing.T) {
	Convey("When a base CPU idle sample is made", t, func() {
		sample := baseCPUSample("idle", "CPU1", float64(5555))
		Convey("Then the sample should units should be count", func() {
			So(sample.Unit, ShouldEqual, telemetry.UnitCount)
		})
		Convey("Then the sample should receive the name 'idle'", func() {
			So(sample.MetricID.Name, ShouldEqual, "idle")
		})
		Convey("Then the sample should receive the dimension for the CPU", func() {
			So(sample.MetricID.Dimensions["CPU"], ShouldEqual, "CPU1")
		})
		Convey("Then the sample value should equal the argument passed in", func() {
			So(sample.Value, ShouldEqual, float64(5555))
		})
	})
}

func TestCPUComputations(t *testing.T) {
	Convey("Given previous and current CPU timing numbers", t, func() {
		lastCPUTimes := cpu.TimesStat{
			CPU:       "CPU1",
			User:      float64(10),
			System:    float64(9),
			Idle:      float64(11),
			Nice:      float64(10),
			Iowait:    float64(12),
			Irq:       float64(11),
			Softirq:   float64(13),
			Steal:     float64(12),
			Guest:     float64(5),
			GuestNice: float64(1),
		}
		curCPUTimes := cpu.TimesStat{
			CPU:       "CPU1",
			User:      float64(20),
			System:    float64(24),
			Idle:      float64(22),
			Nice:      float64(26),
			Iowait:    float64(24),
			Irq:       float64(28),
			Softirq:   float64(26),
			Steal:     float64(30),
			Guest:     float64(8),
			GuestNice: float64(3),
		}

		Convey("When computing differences and totals", func() {
			total := float64(112) / 100
			Convey("Then the total should sum a specific subset of fields", func() {
				So(totalCPUTime(lastCPUTimes), ShouldEqual, float64(88))
				So(totalCPUTime(curCPUTimes), ShouldEqual, float64(200))
			})
			// A bunch of hand-computed diffs here
			Convey("Then the user delta should be a diff of the last and current", func() {
				// This fails tests when checking against 7/total because 6.24999 != 6.25
				So(userDelta(curCPUTimes, lastCPUTimes), ShouldBeBetween, float64(6)/total, float64(8)/total)
			})
			Convey("Then the system delta should be a diff of the last and current", func() {
				So(systemDelta(curCPUTimes, lastCPUTimes), ShouldEqual, float64(15)/total)
			})
			Convey("Then the iowait delta should be a diff of the last and current", func() {
				So(ioWaitDelta(curCPUTimes, lastCPUTimes), ShouldEqual, float64(12)/total)
			})
		})
	})
}
