package agent

import (
	"testing"

	"github.com/shirou/gopsutil/disk"
	. "github.com/smartystreets/goconvey/convey"
)

func TestBaseDiskSample(t *testing.T) {
	Convey("Given a sample created using baseDiskUsageSample", t, func() {
		sample := baseDiskUsageSample("CoolDiskUsageMetric", "/mnt/foobar", float64(1234), "FictionalUnit")
		Convey("Then the name should be correct", func() {
			So(sample.MetricID.Name, ShouldEqual, "CoolDiskUsageMetric")
		})
		Convey("Then the mountpoint should be a dimension", func() {
			So(sample.MetricID.Dimensions["Mountpoint"], ShouldEqual, "/mnt/foobar")
		})
		Convey("Then the value should be correct", func() {
			So(sample.Value, ShouldEqual, float64(1234))
		})
		Convey("Then the unit should be correct", func() {
			So(sample.Unit, ShouldEqual, "FictionalUnit")
		})
	})

	Convey("Given a sample created using baseDiskIOSample", t, func() {
		sample := baseDiskIOSample("CoolDiskIOMetric", "sda", float64(2345), "AnotherFictionalUnit")
		Convey("Then the name should be correct", func() {
			So(sample.MetricID.Name, ShouldEqual, "CoolDiskIOMetric")
		})
		Convey("Then the mountpoint should be a dimension", func() {
			So(sample.MetricID.Dimensions["Device"], ShouldEqual, "sda")
		})
		Convey("Then the value should be correct", func() {
			So(sample.Value, ShouldEqual, float64(2345))
		})
		Convey("Then the unit should be correct", func() {
			So(sample.Unit, ShouldEqual, "AnotherFictionalUnit")
		})
	})
}

func TestDiskIODiffSamples(t *testing.T) {
	Convey("Given 2 consecutive sets of disk IO stats", t, func() {
		lastDiskIOStats := disk.IOCountersStat{
			Name:             "xvda",
			ReadCount:        uint64(1),
			MergedReadCount:  uint64(2),
			WriteCount:       uint64(3),
			MergedWriteCount: uint64(4),
			ReadBytes:        uint64(5),
			WriteBytes:       uint64(6),
			ReadTime:         uint64(7),
			WriteTime:        uint64(8),
			IopsInProgress:   uint64(9),
			IoTime:           uint64(10),
			WeightedIO:       uint64(11),
		}
		curDiskIOStats := disk.IOCountersStat{
			Name:             "xvda",
			ReadCount:        uint64(4),
			MergedReadCount:  uint64(8),
			WriteCount:       uint64(12),
			MergedWriteCount: uint64(16),
			ReadBytes:        uint64(20),
			WriteBytes:       uint64(24),
			ReadTime:         uint64(28),
			WriteTime:        uint64(32),
			IopsInProgress:   uint64(36),
			IoTime:           uint64(40),
			WeightedIO:       uint64(44),
		}
		Convey("When samples are derived from the 2 sets of stats", func() {
			samples := diskIOSamplesFromDiff(curDiskIOStats, lastDiskIOStats)
			Convey("Then there should be exactly 5 samples", func() {
				So(len(samples), ShouldEqual, 5)
			})
			Convey("Then their values should be correct", func() {
				So(samples[0].Value, ShouldEqual, float64(15))
				So(samples[1].Value, ShouldEqual, float64(18))
				So(samples[2].Value, ShouldEqual, float64(0.021))
				So(samples[3].Value, ShouldEqual, float64(0.024))
				So(samples[4].Value, ShouldEqual, float64(0.030))
			})
			Convey("Then their names should be correct, and have the device ID dimension", func() {
				So(samples[0].MetricID.Name, ShouldEqual, "DiskIOReadBytes")
				So(samples[0].MetricID.Dimensions["Device"], ShouldEqual, "xvda")
				So(samples[1].MetricID.Name, ShouldEqual, "DiskIOWriteBytes")
				So(samples[1].MetricID.Dimensions["Device"], ShouldEqual, "xvda")
				So(samples[2].MetricID.Name, ShouldEqual, "DiskIOReadTime")
				So(samples[2].MetricID.Dimensions["Device"], ShouldEqual, "xvda")
				So(samples[3].MetricID.Name, ShouldEqual, "DiskIOWriteTime")
				So(samples[3].MetricID.Dimensions["Device"], ShouldEqual, "xvda")
				So(samples[4].MetricID.Name, ShouldEqual, "DiskIOIoTime")
				So(samples[4].MetricID.Dimensions["Device"], ShouldEqual, "xvda")
			})
		})
	})
}
