package agent

import (
	"testing"

	"github.com/shirou/gopsutil/net"
	. "github.com/smartystreets/goconvey/convey"
)

func TestBaseNetSample(t *testing.T) {
	Convey("Given a sample created using baseNetIfaceSample", t, func() {
		sample := baseNetIfaceSample("CoolNetIfaceMetric", "eth0", float64(1234), "FictionalUnit")
		Convey("Then the name should be correct", func() {
			So(sample.MetricID.Name, ShouldEqual, "CoolNetIfaceMetric")
		})
		Convey("Then there should be a dimension for iface name", func() {
			So(sample.MetricID.Dimensions["Interface"], ShouldEqual, "eth0")
		})
		Convey("Then the unit should be correct", func() {
			So(sample.Unit, ShouldEqual, "FictionalUnit")
		})
		Convey("Then the value should be correct", func() {
			So(sample.Value, ShouldEqual, float64(1234))
		})
	})
}

func TestNetIfaceSamplesFromDiff(t *testing.T) {
	Convey("Given 2 consecutive sets of interface stats", t, func() {
		lastIfaceStats := net.IOCountersStat{
			Name:        "ens3",
			BytesSent:   uint64(0),
			BytesRecv:   uint64(11),
			PacketsSent: uint64(22),
			PacketsRecv: uint64(33),
			Errin:       uint64(44),
			Errout:      uint64(55),
			Dropin:      uint64(66),
			Dropout:     uint64(77),
			Fifoin:      uint64(88),
			Fifoout:     uint64(99),
		}
		curIfaceStats := net.IOCountersStat{
			Name:        "ens3",
			BytesSent:   uint64(24),
			BytesRecv:   uint64(48),
			PacketsSent: uint64(72),
			PacketsRecv: uint64(96),
			Errin:       uint64(120),
			Errout:      uint64(144),
			Dropin:      uint64(168),
			Dropout:     uint64(192),
			Fifoin:      uint64(216),
			Fifoout:     uint64(240),
		}
		Convey("When samples are generated from the 2 sets of stats", func() {
			samples := netIfaceSamplesFromDiff(curIfaceStats, lastIfaceStats)
			Convey("Then there should be exactly 4 samples", func() {
				So(len(samples), ShouldEqual, 4)
			})
			Convey("Then the values should be correct", func() {
				So(samples[0].Value, ShouldEqual, float64(24))
				So(samples[1].Value, ShouldEqual, float64(37))
				So(samples[2].Value, ShouldEqual, float64(50))
				So(samples[3].Value, ShouldEqual, float64(63))
			})
			Convey("Then the interface should be an included dimension", func() {
				So(samples[0].MetricID.Dimensions["Interface"], ShouldEqual, "ens3")
				So(samples[1].MetricID.Dimensions["Interface"], ShouldEqual, "ens3")
				So(samples[2].MetricID.Dimensions["Interface"], ShouldEqual, "ens3")
				So(samples[3].MetricID.Dimensions["Interface"], ShouldEqual, "ens3")
			})
			Convey("Then the names should be correct", func() {
				So(samples[0].MetricID.Name, ShouldEqual, "NetBytesSent")
				So(samples[1].MetricID.Name, ShouldEqual, "NetBytesRecv")
				So(samples[2].MetricID.Name, ShouldEqual, "NetPacketsSent")
				So(samples[3].MetricID.Name, ShouldEqual, "NetPacketsRecv")
			})
		})
	})
}
