package agent

import (
	"fmt"
	"testing"

	logging "code.justin.tv/amzn/TwitchLogging"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	. "github.com/smartystreets/goconvey/convey"
)

type mockLogger struct {
	callCount int
}

func (ml *mockLogger) Log(msg string, keyvals ...interface{}) {
	// increment the number of times this logger was invoked
	ml.callCount++
}

func logger() logging.Logger {
	return &mockLogger{}
}

type mockPollingJob struct{}

// Fetch returns 2 hardcoded samples
func (mpj *mockPollingJob) Fetch() ([]*telemetry.Sample, error) {
	return []*telemetry.Sample{
		&telemetry.Sample{
			MetricID: *telemetry.NewMetricID("FooCount"),
			Value:    float64(5),
			Unit:     telemetry.UnitCount,
		},
		&telemetry.Sample{
			MetricID: *telemetry.NewMetricID("BarSeconds"),
			Value:    float64(7.8),
			Unit:     telemetry.UnitSeconds,
		},
	}, nil

}

type mockFailingPollingJob struct{}

// Fetch always returns an error
func (mfpj *mockFailingPollingJob) Fetch() ([]*telemetry.Sample, error) {
	return nil, fmt.Errorf("mockFailingPollingJob.Fetch always returns an error")
}

func TestRegisterJob(t *testing.T) {
	Convey("Given a basic polling job and a system metrics job harness", t, func() {
		harness := newSystemMetricsPollingJob(logger())
		jobName := "foobar"
		job := &mockPollingJob{}

		Convey("When the mock job is registered to the harness", func() {
			harness.registerJob(jobName, job)

			Convey("Then the job should be in the job map", func() {
				fetchedJob, found := harness.systemMetricPollingJobs[jobName]
				So(harness.jobIsRegistered(jobName), ShouldEqual, true)
				So(found, ShouldEqual, true)
				So(fetchedJob, ShouldEqual, job)
			})

			Convey("And the same job is registered to a different name", func() {
				harness.registerJob("foobar2", job)

				Convey("Then there will be two jobs in the harness", func() {
					fetchedJob2, found2 := harness.systemMetricPollingJobs["foobar2"]
					So(harness.jobIsRegistered("foobar2"), ShouldEqual, true)
					So(found2, ShouldEqual, true)
					So(fetchedJob2, ShouldEqual, job)
				})
			})
		})

	})
}

func TestFetch(t *testing.T) {
	Convey("Given a basic polling job and a system metrics job harness", t, func() {
		harness := newSystemMetricsPollingJob(logger())
		jobName := "foobar"
		job := &mockPollingJob{}

		Convey("When the job is registered to the harness", func() {
			harness.registerJob(jobName, job)

			Convey("And samples are fetched", func() {
				samples, err := harness.Fetch()

				Convey("Then there should be no errors", func() {
					So(err, ShouldBeNil)
				})

				Convey("Then there should be exactly 2 samples", func() {
					So(len(samples), ShouldEqual, 2)
				})
			})

			Convey("And the job is registered to the harness 4 more times", func() {
				harness.registerJob("foobar2", job)
				harness.registerJob("foobar3", job)
				harness.registerJob("foobar4", job)
				harness.registerJob("foobar5", job)

				Convey("And samples are fetched", func() {
					samples, err := harness.Fetch()

					Convey("Then there should be no errors", func() {
						So(err, ShouldBeNil)
					})

					Convey("Then there should be exactly 10 samples", func() {
						So(len(samples), ShouldEqual, 10)
					})
				})
			})

			Convey("And a failing job is also registered to the harness", func() {
				failingJob := &mockFailingPollingJob{}
				harness.registerJob("failJob", failingJob)

				Convey("When samples are fetched", func() {
					samples, err := harness.Fetch()

					Convey("Then there should still be no errors (they get swallowed)", func() {
						So(err, ShouldBeNil)
					})

					Convey("Then there should still be 2 samples returned", func() {
						So(len(samples), ShouldEqual, 2)
					})

					Convey("Then exactly one message gets submitted to the underlying logger", func() {
						So(harness.logger.(*mockLogger).callCount, ShouldEqual, 1)
					})
				})
			})
		})
	})
}
