package main

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"strings"

	"github.com/pkg/errors"
	"gopkg.in/urfave/cli.v1"
)

// TwitchTitleCliConfig is a struct used to hold information for the translation CLI
type TwitchTitleCliConfig struct {
	// SmartlingUserID is the smartling user identifier -- this can be found in the Smartling API V2.0 Tokens
	SmartlingUserID string

	// SmartlingUserID is the smartling project identifier -- this can be found in the Smartling API V2.0 Tokens
	SmartlingProjectID string

	// SmartlingFileURI is where your uploaded file will be stored in smartling
	SmartlingFileURI string

	// AwsAccountID is the ID of an AwsAccount that can access the sandstorm secret
	AwsAccountID string

	// AwsAccountID is the role in the above AwsAccount that can access the sandstorm secret
	AwsAccountRole string

	// SandstormRoleArn is the sandstorm role that is used to access your smartling secret token
	SandstormRoleArn string

	// SandstormSecretTokenName is the name of the sandstorm secret that holds your smartling secret token
	SandstormSecretTokenName string
}

func LoadConfig(c *cli.Context) (*TwitchTitleCliConfig, error) {
	// Read the configruation json file
	configBytes, err := ioutil.ReadFile(configJsonUri)
	if err != nil {
		return nil, cli.NewExitError(fmt.Sprintf("Failed to read the configuration file: %s", err.Error()), 1)
	}
	var config TwitchTitleCliConfig
	err = json.Unmarshal(configBytes, &config)
	if err != nil {
		return nil, cli.NewExitError(fmt.Sprintf("Failed to unmarshal the json of the configruation file: %s", err.Error()), 1)
	}

	// Validate the configuration file parameters
	if err = config.validate(); err != nil {
		return nil, cli.NewExitError(fmt.Sprintf("Invalid configruation file: %s", err.Error()), 1)
	}

	return &config, nil
}

func (c *TwitchTitleCliConfig) validate() error {
	// Validate that all parameters are set
	if err := c.validateParamsAreSet(); err != nil {
		return err
	}

	// Validate that all paths are valid
	if err := c.validatePathsAreValid(); err != nil {
		return err
	}

	// Validate that the smartling secret token can be retrieved from sandstorm
	_, err := GetSandstormSecret(c.AwsAccountID, c.AwsAccountRole, c.SandstormRoleArn, c.SandstormSecretTokenName)
	if err != nil {
		return errors.Wrap(err, "failed to fetch sandstorm secret")
	}

	// Check that the config file is an unmarshalable json file
	uploadFileBytes, err := ioutil.ReadFile(configJsonUri)
	if err != nil {
		return errors.Wrap(err, "failed to read the titles file")
	}
	var result map[string]interface{}
	err = json.Unmarshal(uploadFileBytes, &result)
	if err != nil {
		return errors.Wrap(err, "failed to unmarshal the json of the titles file")
	}

	return nil
}

func (c *TwitchTitleCliConfig) validateParamsAreSet() error {
	if c.SmartlingUserID == "" {
		return errors.New("SmartlingUserID not set")
	}
	if c.SmartlingProjectID == "" {
		return errors.New("SmartlingProjectID not set")
	}
	if c.SmartlingFileURI == "" {
		return errors.New("SmartlingFileURI not set")
	}
	if c.AwsAccountID == "" {
		return errors.New("AwsAccountID not set")
	}
	if c.AwsAccountRole == "" {
		return errors.New("AwsAccountRole not set")
	}
	if c.SandstormRoleArn == "" {
		return errors.New("SandstormRoleArn not set")
	}
	if c.SandstormSecretTokenName == "" {
		return errors.New("SandstormSecretTokenName not set")
	}
	return nil
}

func (c *TwitchTitleCliConfig) validatePathsAreValid() error {
	// Check types for files that do not exist yet
	if err := checkFileType(c.SmartlingFileURI, "json"); err != nil {
		return errors.Wrap(err, "SmartlingFileURI is invalid")
	}
	if err := checkFileType(generatedTitleMapURI, "go"); err != nil {
		return errors.Wrap(err, fmt.Sprintf("%s must be a go file!", generatedTitleMapURI))
	}

	// Check that the URIs from the `init` command have been set up
	if err := checkPathIsValid(configJsonUri, "json"); err != nil {
		return errors.Wrap(err, fmt.Sprintf("%s does not exist!", configJsonUri))
	}
	if err := checkPathIsValid(titlesJsonUri, ""); err != nil {
		return errors.Wrap(err, fmt.Sprintf("%s does not exist!", titlesJsonUri))
	}
	if err := checkPathIsValid(translationsFolderUri, ""); err != nil {
		return errors.Wrap(err, fmt.Sprintf("%s does not exist!", translationsFolderUri))
	}
	return nil
}

func checkPathIsValid(path, fileType string) error {
	// Check that the path exists
	exists, err := pathExists(path)
	if !exists {
		return errors.New("path does not exist")
	}
	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("error checking if path is valid"))
	}

	// Check that the file type is valid
	if err = checkFileType(path, fileType); err != nil {
		return err
	}
	return nil
}

func checkFileType(path, fileType string) error {
	// If this is a file, check the file type
	if fileType != "" {
		splitPath := strings.Split(path, ".")
		if splitPath[len(splitPath)-1] != fileType {
			return errors.New(fmt.Sprintf("path not of type %s", fileType))
		}
	}
	return nil
}

func pathExists(path string) (bool, error) {
	_, err := os.Stat(path)
	if err == nil {
		return true, nil
	}
	if os.IsNotExist(err) {
		return false, nil
	}
	return true, err
}
