package main

import (
	"fmt"
	"io/ioutil"
	"path"

	titles "code.justin.tv/amzn/TwitchTitles"

	smartling "github.com/Smartling/api-sdk-go"
	"gopkg.in/urfave/cli.v1"
)

// languageSubstitutes are languages not supported by Smartling with the proper substitute to use
var languageSubstitutes = map[titles.LanguageCode]titles.LanguageCode{
	"en-GB": "en-US", // 'en-GB' is not supported, use 'en-US' instead
}

func downloadCommand() cli.Command {
	return cli.Command{
		Name:      "download",
		ShortName: "dn",
		Usage:     "This will download a translation file from Smartling",
		Action:    downloadAction,
	}
}

func downloadAction(c *cli.Context) error {
	// Load in the config from the cli context
	config, err := LoadConfig(c)
	if err != nil {
		return err
	}

	// Get the translation file bytes for every supported language (map[language] --> bytes)
	translationFileMap, err := downloadTranslations(config, titles.SupportedLanguages)
	if err != nil {
		return cli.NewExitError(fmt.Sprintf("failed to download translations: %s", err.Error()), 1)
	}

	// Update the local translation files
	err = updateLocalTranslations(config, translationFileMap)
	if err != nil {
		return cli.NewExitError(fmt.Sprintf("failed to update local translations: %s", err.Error()), 1)
	}

	fmt.Println("Done download!")

	return nil
}

func downloadTranslations(c *TwitchTitleCliConfig, languageIDs []titles.LanguageCode) (map[titles.LanguageCode]RawTranslationData, error) {
	// Create a Smartling API client
	secretToken, err := GetSandstormSecret(c.AwsAccountID, c.AwsAccountRole, c.SandstormRoleArn, c.SandstormSecretTokenName)
	if err != nil {
		return nil, err
	}
	smartlingApiClient := smartling.NewClient(c.SmartlingUserID, secretToken)

	// Build the download request
	downloadRequest := smartling.FileDownloadRequest{
		FileURIRequest: smartling.FileURIRequest{
			FileURI: c.SmartlingFileURI,
		},
		Type:            "published",
		IncludeOriginal: false,
	}

	// Download and store all translations to the translationFileMap
	translationFileMap := make(map[titles.LanguageCode]RawTranslationData)
	for _, langCode := range titles.SupportedLanguages {
		if _, exists := languageSubstitutes[langCode]; exists {
			continue
		}
		reader, err := smartlingApiClient.DownloadTranslation(c.SmartlingProjectID, langCode.String(), downloadRequest)
		if err != nil {
			return nil, err
		}
		bytes, err := ioutil.ReadAll(reader)
		if err != nil {
			return nil, err
		}
		translationFileMap[langCode], err = UnmarshalTranslationFile(bytes)
		if err != nil {
			return nil, err
		}
	}

	// Insert all locale subsititues into the translationFileMap
	for originalCode, substituteCode := range languageSubstitutes {
		translationFileMap[originalCode] = translationFileMap[substituteCode]
	}

	fmt.Println(fmt.Sprintf("Downloaded translations for %d locales from Smartling", len(translationFileMap)))

	return translationFileMap, nil
}

func updateLocalTranslations(config *TwitchTitleCliConfig, translationFileMap map[titles.LanguageCode]RawTranslationData) error {
	// Write the bytes of each downloaded translation file
	for languageID, translationData := range translationFileMap {
		fileName := languageID.String() + ".json"
		filePath := path.Join(translationsFolderUri, fileName)
		bytes, err := MarshalTranslationData(translationData)
		if err != nil {
			return err
		}
		err = ioutil.WriteFile(filePath, bytes, fileWriteMode)
		if err != nil {
			return err
		}
	}

	// Write all of the translations to a map
	titleMap := generateLocalizedTitleMap(translationFileMap)
	err := ioutil.WriteFile(generatedTitleMapURI, titleMap, fileWriteMode)
	if err != nil {
		return err
	}

	return nil
}
