package main

import (
	"fmt"
	"sort"
	"strings"

	titles "code.justin.tv/amzn/TwitchTitles"
	"code.justin.tv/amzn/TwitchTitles/tokenizer"
)

// Generate a header for generated translation files
func generateFileHeader() string {
	return strings.Join([]string{
		"// Code generated by the twitch-title-cli. DO NOT EDIT.",
		"package localization",
		"",
		"import (",
		"\ttitles \"code.justin.tv/amzn/TwitchTitles\"",
		"\t\"code.justin.tv/amzn/TwitchTitles/tokenizer\"",
		")",
		"",
	}, "\n")
}

func generateLocalizedTitleMap(translationFileMap map[titles.LanguageCode]RawTranslationData) []byte {
	// Build the header of the localized title map
	titleMap := strings.Join([]string{
		generateFileHeader(),
		"// LocalizedTitleMap contains localized translations for every supported language.",
		"var LocalizedTitleMap = titles.TitleMap {",
		generateTitlesForLanguages(translationFileMap),
		"}",
		"",
	}, "\n")

	return []byte(titleMap)
}

func generateTitlesForLanguages(translationFileMap map[titles.LanguageCode]RawTranslationData) string {
	var titlesForLanguages []string
	for languageID, data := range translationFileMap {
		titlesForLanguages = append(titlesForLanguages, strings.Join([]string{
			fmt.Sprintf("\t%q: titles.TitleData {", languageID.String()),
			generateTitleDataForLanguage(data),
			"\t},",
		}, "\n"))
	}

	sort.Strings(titlesForLanguages)

	return strings.Join(titlesForLanguages, "\n")
}

func generateTitleDataForLanguage(translationData RawTranslationData) string {
	var titleDataForLanguage []string
	for _, title := range translationData.Titles {
		titleDataForLanguage = append(titleDataForLanguage, strings.Join([]string{
			fmt.Sprintf("\t\t%q: titles.Title {", title.Name),
			generateTitle(title),
			"\t\t},",
		}, "\n"))
	}

	sort.Strings(titleDataForLanguage)

	return strings.Join(titleDataForLanguage, "\n")
}

func generateTitle(data RawTitleData) string {
	var variantTitles []string
	for variant, title := range data.TitleVariants {
		variantTokenMap, err := tokenizer.GetTitleTokens(title, variant)
		if err != nil {
			panic(fmt.Sprintf("Failed to tokenize a title: %q", title))
		}

		for generatedVariant, tokens := range variantTokenMap {
			variantTitles = append(variantTitles, strings.Join([]string{
				fmt.Sprintf("\t\t\t\t%q: []tokenizer.TitleToken{", generatedVariant),
				generateTitleTokens(tokens),
				"\t\t\t\t},",
			}, "\n"))
		}
	}

	sort.Strings(variantTitles)

	return strings.Join([]string{
		fmt.Sprintf("\t\t\tFallbackTitle: %q,", data.FallbackTitle),
		fmt.Sprintf("\t\t\tTitleVariants: titles.TitleVariantMap {"),
		strings.Join(variantTitles, "\n"),
		"\t\t\t},",
	}, "\n")
}

func generateTitleTokens(tokens []tokenizer.TitleToken) string {
	var tokenStrings []string
	for _, token := range tokens {
		var tagString string
		if len(token.Tags) == 0 {
			tagString = "[]string{}"
		} else {
			tagString = strings.Join(token.Tags, `", "`)
			tagString = fmt.Sprintf("[]string{\"%s\"}", tagString)
		}

		tokenStrings = append(tokenStrings, strings.Join([]string{
			"\t\t\t\t\t{",
			fmt.Sprintf("\t\t\t\t\t\tType: tokenizer.%v,", token.Type),
			fmt.Sprintf("\t\t\t\t\t\tText: %q,", token.Text),
			fmt.Sprintf("\t\t\t\t\t\tTags: %v,", tagString),
			"\t\t\t\t\t},",
		}, "\n"))
	}

	return strings.Join(tokenStrings, "\n")
}
