package main

import (
	"time"

	"code.justin.tv/systems/sandstorm/manager"

	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/pkg/errors"

	"golang.a2z.com/GoAmzn-AmazonCACerts/amazoncacerts"
	"golang.a2z.com/GoAmzn-IsengardProvider/isengardcreds"
	"golang.a2z.com/GoAmzn-MidwayCookie/cookiefile"
)

// GetSandstormSecret fetches a sandstorm secret using isengard_credentials (https://git-aws.internal.justin.tv/twitch/isengard_credentials)
func GetSandstormSecret(awsAccountID, awsAccountRole, sandstormRoleArn, sandstormSecretName string) (string, error) {
	cf, err := cookiefile.New()
	if err != nil {
		return "", errors.Wrap(err, "failed to create new cookie file")
	}

	// Create an IsengardProvider
	client := amazoncacerts.GetHttpClient(false)
	client.Jar = cf.Jar
	client.Timeout = 10 * time.Second

	provider, err := isengardcreds.New(awsAccountID, awsAccountRole, client)
	if err != nil {
		return "", errors.Wrap(err, "error creating isengard provider")
	}

	// Use the IsengardProvider as the credentials provider for you AWS SDK
	// services.
	creds := credentials.NewCredentials(provider)

	managerAWSConfig := manager.AWSConfig(creds, sandstormRoleArn)

	m := manager.New(manager.Config{
		AWSConfig: managerAWSConfig,
	})

	secret, err := m.Get(sandstormSecretName)
	if err != nil {
		return "", errors.Wrap(err, "failed to fetch secret from sandstorm")
	}
	if secret == nil {
		return "", errors.New("secret does not exist")
	}
	return string(secret.Plaintext), nil
}
