package titles

import (
	"code.justin.tv/amzn/TwitchTitles/tokenizer"
)

// Valid plural keys
const (
	ZeroPluralKey  = "zero"
	OnePluralKey   = "one"
	TwoPluralKey   = "two"
	FewPluralKey   = "few"
	ManyPluralKey  = "many"
	OtherPluralKey = "other"
)

func getLanguagePluralKey(number int64, variant TitleVariant, languageCode LanguageCode) TitleVariant {
	oneVariant := TitleVariant(tokenizer.FormatPluralVariant(string(variant), OnePluralKey))
	fewVariant := TitleVariant(tokenizer.FormatPluralVariant(string(variant), FewPluralKey))
	manyVariant := TitleVariant(tokenizer.FormatPluralVariant(string(variant), ManyPluralKey))
	otherVariant := TitleVariant(tokenizer.FormatPluralVariant(string(variant), OtherPluralKey))

	switch languageCode {
	case DefaultLanguage:
		fallthrough
	case "bg-BG":
		fallthrough
	case "da-DK":
		fallthrough
	case "de-DE":
		fallthrough
	case "el-GR":
		fallthrough
	case "en-GB":
		fallthrough
	case "es-MX":
		fallthrough
	case "es-ES":
		fallthrough
	case "fi-FI":
		fallthrough
	case "hu-HU":
		fallthrough
	case "it-IT":
		fallthrough
	case "nl-NL":
		fallthrough
	case "no-NO":
		fallthrough
	case "pt-PT":
		fallthrough
	case "sv-SE":
		fallthrough
	case "tr-TR":
		if number == 1 {
			return oneVariant
		}
		return otherVariant

	case "cs-CZ":
		fallthrough
	case "sk-SK":
		if number == 1 {
			return oneVariant
		} else if number >= 2 && number <= 4 {
			return fewVariant
		}
		return otherVariant

	case "fr-FR":
		fallthrough
	case "pt-BR":
		if number == 0 || number == 1 {
			return oneVariant
		}
		return otherVariant

	case "pl-PL":
		lastDigit := number % 10
		if number == 1 {
			return oneVariant
		} else if lastDigit >= 2 && lastDigit <= 4 && number != 12 && number != 13 && number != 14 {
			return fewVariant
		}
		return manyVariant

	case "ro-RO":
		if number == 1 {
			return oneVariant
		} else if number == 0 || number%100 < 20 {
			return fewVariant
		}
		return otherVariant

	case "ru-RU":
		lastDigit := number % 10
		if lastDigit == 1 && number != 11 {
			return oneVariant
		} else if lastDigit >= 2 && lastDigit <= 4 && number != 12 && number != 13 && number != 14 {
			return fewVariant
		}
		return manyVariant

	case "ja-JP":
		fallthrough
	case "ko-KR":
		fallthrough
	case "th-TH":
		fallthrough
	case "vi-VN":
		fallthrough
	case "zh-TW":
		fallthrough
	case "zh-CN":
		return otherVariant

	default:
		return variant
	}
}
