package titles

import (
	"regexp"

	"golang.org/x/text/language"
)

// LanguageCode represents the language code for a specific region
type LanguageCode string

func (l LanguageCode) String() string {
	return string(l)
}

// DefaultLanguage is the defualt language to use when handling requests
const DefaultLanguage LanguageCode = "en-US"

// SupportedLanguages contains all of the languages that TwitchTitles team currently supports.
// The first element in this list is returned if the languageMatcher fails to match a language.
var SupportedLanguages = []LanguageCode{
	DefaultLanguage, // this is "en-US", or the default language
	"bg-BG",
	"cs-CZ",
	"da-DK",
	"de-DE",
	"el-GR",
	"en-GB",
	"es-MX",
	"es-ES",
	"fi-FI",
	"fr-FR",
	"hu-HU",
	"it-IT",
	"ja-JP",
	"ko-KR",
	"nl-NL",
	"no-NO",
	"pl-PL",
	"pt-BR",
	"pt-PT",
	"ro-RO",
	"ru-RU",
	"sk-SK",
	"sv-SE",
	"th-TH",
	"tr-TR",
	"vi-VN",
	"zh-TW",
	"zh-CN",
}

// Singleton instance of a language matcher
var languageMatcher *language.Matcher

// StatSafeString converts a LanguageCode into a standard string for use in stats
func (l LanguageCode) StatSafeString() string {
	return regexp.MustCompile(`[^A-Za-z0-9-]+`).ReplaceAllString(l.String(), "-")
}

// getLanguageToServe returns the language code for the first language that is supported.
func getLanguageToServe(languages []string) LanguageCode {
	_, supportedIndex := language.MatchStrings(*getLanguageMatcher(), languages...)
	return SupportedLanguages[supportedIndex]
}

// getLanguageMatcher returns a language matcher for Twitch's supported languages
func getLanguageMatcher() *language.Matcher {
	if languageMatcher == nil {
		var tags []language.Tag
		for _, languageCode := range SupportedLanguages {
			tags = append(tags, language.MustParse(string(languageCode)))
		}

		matcher := language.NewMatcher(tags)
		languageMatcher = &matcher
	}

	return languageMatcher
}
