package titles

import (
	"fmt"

	"code.justin.tv/amzn/TwitchTitles/tokenizer"
	"github.com/pkg/errors"
)

// TitleMap contains all of the titles available in the generated map
type TitleMap map[LanguageCode]TitleData

// TitleData contains all of the titles for a specific language
type TitleData map[string]Title

// Title defines the fallback title if tokenization fails and a set of tokens for different variants
type Title struct {
	FallbackTitle string
	TitleVariants TitleVariantMap
}

// TitleMapGetParams are params used to get a title from this library
type TitleMapGetParams struct {
	Languages         []string
	TitleName         string
	VariantName       TitleVariant
	TitlePlaceholders PlaceholderMap
}

// TitleVariantMap is a map of title variants to tokens
type TitleVariantMap map[TitleVariant][]tokenizer.TitleToken

// TitleVariant is the type used to define a variant
type TitleVariant string

// PlaceholderMap is a map of placeholder types to placeholder IDs to insert
type PlaceholderMap map[PlaceholderType][]TwitchTitleTokenPlaceholder

// PlaceholderType defines the type of placeholder that will be inserted
type PlaceholderType string

const (
	// The DefaultTitleVariant to find in the titleMap
	DefaultTitleVariant TitleVariant = "Default"

	// UnknownPlaceholder is a placeholder type
	UnknownPlaceholder PlaceholderType = "UNKNOWN_PLACEHOLDER"

	// StreamPlaceholder is a placeholder that contains a StreamID
	StreamPlaceholder PlaceholderType = "STREAM_PLACEHOLDER"

	// CategoryPlaceholder is a placeholder thast ocntains a CategoryID
	CategoryPlaceholder PlaceholderType = "CATEGORY_PLACEHOLDER"

	// TagPlaceholder is a placeholder that contains a TagID
	TagPlaceholder PlaceholderType = "TAG_PLACEHOLDER"

	// NumberPlaceholder is a placeholder that contains a Number
	NumberPlaceholder PlaceholderType = "NUMBER_PLACEHOLDER"

	// DatePlaceholder is a placeholder that contains a Date
	DatePlaceholder PlaceholderType = "DATE_PLACEHOLDER"
)

// TwitchTitle is the type that will be returned to users of this library
type TwitchTitle struct {
	Name                   string
	FallbackLocalizedTitle string
	LocalizedTitleTokens   []*TwitchTitleToken
}

// TwitchTitleTokenType represents a type of a twitch title token
type TwitchTitleTokenType int

type TwitchTitleTextTokenHyperlink int

// TwitchTitleToken is a segment of the title that allows for complex title logic
type TwitchTitleToken struct {
	TokenType        TwitchTitleTokenType
	TextToken        *TwitchTitleTokenText
	PlaceholderToken *TwitchTitleTokenPlaceholder
}

// TwitchTitleTokenPlaceholder is a token that is a placeholder
type TwitchTitleTokenPlaceholder struct {
	// Category ID, Stream ID, or Tag ID
	ID string

	// Used for NumberPlaceholders and epoch time for DatePlaceholders
	Value int64
}

// TwitchTitleTokenText is a token that is a piece of text
type TwitchTitleTokenText struct {
	Text     string
	Location TwitchTitleTextTokenHyperlink
	Emphasis bool
}

const (
	NO_LINK TwitchTitleTextTokenHyperlink = iota
	CATEGORY_DIRECTORY_LINK
	STREAMS_DIRECTORY_LINK
)

const (
	UNKNOWN_TOKEN_TYPE TwitchTitleTokenType = iota
	TEXT_TOKEN_TYPE
	CATEGORY_TOKEN_TYPE
	STREAM_TOKEN_TYPE
	TAG_TOKEN_TYPE
	NUMBER_TOKEN_TYPE
	DATE_TOKEN_TYPE
)

func (p *TitleMapGetParams) validate() error {
	if p == nil {
		return errors.New("received nil TitleMapGetParams")
	}

	if len(p.Languages) < 1 {
		return errors.New("must have at least 1 language in TitleMapGetParams")
	}

	if p.TitleName == "" {
		return errors.New("must specify a TitleName in TitleMapGetParams")
	}

	return nil
}

// GetTitle returns a titleTwirp title from a TitleMap, using input parameters
func (m TitleMap) GetTitle(params *TitleMapGetParams) (*TwitchTitle, error) {
	// Validate the get params
	if err := params.validate(); err != nil {
		return nil, errors.Wrap(err, "invalid get from titleMap")
	}

	// Get the titleData for the closest language
	languageCode := getLanguageToServe(params.Languages)
	titleData, ok := m[languageCode]
	if !ok {
		return nil, errors.New(fmt.Sprintf("could not find language code %s in titleMap", languageCode))
	}

	// Get the title from that titleName
	title, ok := titleData[params.TitleName]
	if !ok {
		return nil, errors.New(fmt.Sprintf("could not find titleName for language code %s: %v", languageCode, *params))
	}

	// Get the fallback title string from the  title
	fallbackTitle := title.FallbackTitle

	// Get the title tokens from the title
	var titleTokens []tokenizer.TitleToken
	variantName := DefaultTitleVariant
	if params.VariantName != "" {
		variantName = params.VariantName
	}
	// Check for number placeholder for plural title variant
	numberPlaceholder, ok := params.TitlePlaceholders[NumberPlaceholder]
	if ok {
		// plural syntax does not work for more than one number placeholder, so we are guaranteed
		// that the placeholder must be index
		variantName = getLanguagePluralKey(numberPlaceholder[0].Value, variantName, languageCode)
	}

	// If a variant is passed in, get the tokens for that variant
	titleTokens, ok = title.TitleVariants[variantName]
	if !ok {
		return &TwitchTitle{
			Name:                   params.TitleName,
			FallbackLocalizedTitle: fallbackTitle,
		}, errors.New(fmt.Sprintf("could not find titleVariant %s for language code %s. if the title is a plural, it may not be translated yet", variantName, languageCode))
	}

	localizedTitleTokens, err := buildResponseTokens(titleTokens, params.TitlePlaceholders)
	if err != nil {
		return nil, errors.Wrap(err, "failed to build title tokens")
	}

	return &TwitchTitle{
		Name:                   params.TitleName,
		FallbackLocalizedTitle: fallbackTitle,
		LocalizedTitleTokens:   localizedTitleTokens,
	}, nil
}
