package tokenizer

import (
	"fmt"
	"unicode"
	"unicode/utf8"
)

const EOF = -1

// TitleTokenType defines the type of title tokens the lexer will output
type TitleTokenType int

// Types of tokens that will be emitted by the lexer
const (
	TextToken = TitleTokenType(iota)
	PlaceholderToken
)

// TitleToken defines a token object that will be emitted
type TitleToken struct {
	Type TitleTokenType
	Text string
	Tags []string
}

// String returns the type of a title token in string form
func (t TitleTokenType) String() string {
	switch t {
	case TextToken:
		return "TextToken"
	case PlaceholderToken:
		return "PlaceholderToken"
	default:
		return fmt.Sprintf("%d", int(t))
	}
}

// Constants that change the state of the lexer
const (
	tagStart         = '<'
	tagEndSignal     = '/'
	tagEnd           = '>'
	placeholderStart = '{'
	placeholderEnd   = '}'
	pluralStart      = ','
	pluralNextSignal = ' '
)

// isValidChar types take in a rune and returns if it is valid in a specific context
type isValidRune func(rune) bool

// isValidInTag returns true if a rune can be in a tag
func isValidInTag(r rune) bool {
	return unicode.IsLetter(r) || r == ':'
}

// isValidInPlaceholder returns true if a rune can be in a placeholder
func isValidInPlaceholder(r rune) bool {
	return unicode.IsLetter(r)
}

// GetTitleTokens gets a mapping of variants to TitleTokens from a title and variant
func GetTitleTokens(title string, variant string) (map[string][]TitleToken, error) {
	l := &lexer{
		input:   title,
		variant: variant,
	}
	l.run()

	return l.items, l.err
}

// lexer holds the state of the scanner
type lexer struct {
	input          string                  // the string being scanned
	start          int                     // start position of this title token
	pos            int                     // current position in the input
	width          int                     // width of last rune read from input
	tags           []string                // current tags active for this title token
	isPluralTitle  bool                    // whether the text is currently in a plural title
	variant        string                  // variant to generate tokens for
	currentVariant string                  // current variant
	items          map[string][]TitleToken // map of variant names to scanned title tokens
	err            error                   // an error set if an error is encountered
}

// stateFn represents the state of the scanner as a function that returns the next state
type stateFn func(*lexer) stateFn

// run lexes the input by executing state functions until the state is nil
func (l *lexer) run() {
	l.items = make(map[string][]TitleToken)
	for state := lexText; state != nil; {
		if l.err != nil {
			break
		}
		state = state(l)
	}
}

// lexText is the main state function, it handles Text
func lexText(l *lexer) stateFn {
	for {
		// Handle incoming tags and placeholders
		if l.beginsWithRune(tagStart) {
			if l.pos > l.start {
				l.emit(TextToken)
			}
			return lexOpeningTag
		} else if l.beginsWithRune(placeholderStart) {
			if l.pos > l.start {
				l.emit(TextToken)
			}
			return lexPlaceholder
		} else if l.beginsWithRune(placeholderEnd) {
			if l.isPluralTitle && l.pos > l.start {
				l.emit(TextToken)
			} else {
				l.isPluralTitle = false
			}
			l.skip(placeholderEnd)
			return lexPluralAddVariant
		}
		if l.next() == EOF {
			break
		}
	}
	// Correctly reached EOF
	if l.pos > l.start {
		l.emit(TextToken)
	}
	// Emit an error if all tags were not closed
	if len(l.tags) > 0 {
		l.errorf("not all tags have been closed")
	}
	// Stop the run loop
	return nil
}

// lexOpeningTag is a state function that runs when we see a tag
func lexOpeningTag(l *lexer) stateFn {
	l.skip(tagStart)
	if l.peek() == tagEndSignal {
		l.skip(tagEndSignal)
		return lexRemoveTag
	}
	return lexAddTag
}

// lexAddTag is a state function that runs when we want to add a tag
func lexAddTag(l *lexer) stateFn {
	l.acceptUntil(isValidInTag, tagEnd)
	l.addTag()
	l.skip(tagEnd)
	return lexText
}

// lexRemoveTag is a state function that runs when we want to remove a tag
func lexRemoveTag(l *lexer) stateFn {
	l.acceptUntil(isValidInTag, tagEnd)
	if !l.removeTag() {
		return l.errorf("failed to remove tag")
	}
	l.skip(tagEnd)
	return lexText
}

// lexPlaceholder is a state function that emits a placeholder
func lexPlaceholder(l *lexer) stateFn {
	l.skip(placeholderStart)
	l.acceptUntil(isValidInPlaceholder, placeholderEnd, pluralStart)
	if l.beginsWithRune(pluralStart) {
		l.skip(pluralStart)
		return lexPluralOpening
	}
	l.emit(PlaceholderToken)
	l.skip(placeholderEnd)
	return lexText
}

// lexPluralOpening is a state function that runs when we identify that the
// title is a plural title, and we need to extract separate plural strings from
// the title
func lexPluralOpening(l *lexer) stateFn {
	l.skip(pluralNextSignal)
	l.acceptUntil(isValidInPlaceholder, pluralStart)
	l.skip(pluralStart)
	return lexPluralAddVariant
}

// lexPluralAddVariant is a state function that runs when we want to add
func lexPluralAddVariant(l *lexer) stateFn {
	// the next character might be a closing parenthesis, indicating that there
	// are no more plural forms, so we can go back to lexText finish
	if l.peek() == placeholderEnd {
		l.skip(placeholderEnd)
		return lexText
	}
	l.isPluralTitle = true
	l.skip(pluralNextSignal)
	// set the variant based on the plural amount
	l.acceptUntil(isValidInPlaceholder, pluralNextSignal)
	l.currentVariant = FormatPluralVariant(l.variant, l.input[l.start:l.pos])

	l.skip(pluralNextSignal)
	l.skip(placeholderStart)
	return lexText
}

// next returns the next rune in the input
func (l *lexer) next() (val rune) {
	if l.pos >= len(l.input) {
		l.width = 0
		return EOF
	}
	val, l.width = utf8.DecodeRuneInString(l.input[l.pos:])
	l.pos += l.width
	return val
}

// peek returns but does not consume the next rune in the input
func (l *lexer) peek() rune {
	val := l.next()
	l.backup()
	return val
}

// ignore skips over the pending input before this point
func (l *lexer) ignore() {
	l.start = l.pos
}

// backup steps back one rune (can be called only once per call of next)
func (l *lexer) backup() {
	l.pos -= l.width
}

// skip checks that the next rune is anticipated and skips it if so
func (l *lexer) skip(expected rune) {
	if !l.beginsWithRune(expected) {
		l.errorf("lexer states incorrectly joined, tried to skip unexpected rune")
	}
	l.pos++
	l.ignore()
}

// errorf emits an error token that the client will receive
func (l *lexer) errorf(format string, args ...interface{}) stateFn {
	l.err = fmt.Errorf("error in tokenizer, position = %d, input = %q", l.pos, l.input)
	return nil
}

// emit passes an item back to the client
func (l *lexer) emit(t TitleTokenType) {
	// specify the variant
	var variant = l.variant
	if l.currentVariant != "" {
		variant = l.currentVariant
	}

	currentTags := make([]string, len(l.tags))
	copy(currentTags, l.tags)
	l.items[variant] = append(l.items[variant], TitleToken{t, l.input[l.start:l.pos], currentTags})
	l.start = l.pos
}

// acceptUntil accepts all runes until a delimiter or an invalid rune is found
func (l *lexer) acceptUntil(isValid isValidRune, delimiters ...rune) {
	for {
		for _, d := range delimiters {
			if l.beginsWithRune(d) {
				return
			}
		}
		switch r := l.next(); {
		case isValid(r):
			continue
		default:
			l.errorf("invalid rune found")
			return
		}
	}
}

// addTag adds the current item as a tag
func (l *lexer) addTag() {
	tagToAdd := l.input[l.start:l.pos]
	l.tags = append(l.tags, tagToAdd)
}

// removeTag removes the current item as a tag, returning if it successfully removed the item
func (l *lexer) removeTag() bool {
	tagToDelete := l.input[l.start:l.pos]
	for i, tag := range l.tags {
		if tag == tagToDelete {
			l.tags = append(l.tags[:i], l.tags[i+1:]...)
			return true
		}
	}
	return false
}

// beginsWithRune returns true if the next rune to be seen by the lexer is a specific one
func (l *lexer) beginsWithRune(r rune) bool {
	if l.pos >= len(l.input) {
		return false
	}
	return rune(l.input[l.pos]) == r
}

// Formats the variant and pluralKey into the variant key
func FormatPluralVariant(variant string, pluralKey string) string {
	return fmt.Sprintf("%s::%s", variant, pluralKey)
}
