package tokenizer

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
)

var noTags = []string{}
var emphasisTag = []string{"em"}
var linkTag = []string{"link:directory"}
var linkAndEmphasisTag = []string{"link:directory", "em"}

var tokenizerTests = []struct {
	name   string
	title  string
	tokens []TitleToken
}{
	{
		name:  "Basic Tag 1",
		title: "Popular <em>live channels</em>",
		tokens: []TitleToken{
			{Type: TextToken, Text: "Popular ", Tags: noTags},
			{Type: TextToken, Text: "live channels", Tags: emphasisTag},
		},
	},
	{
		name:  "Basic Tag 2",
		title: "<em>Videos</em> recommended for you",
		tokens: []TitleToken{
			{Type: TextToken, Text: "Videos", Tags: emphasisTag},
			{Type: TextToken, Text: " recommended for you", Tags: noTags},
		},
	},
	{
		name:  "Full Tag 1",
		title: "<em>Popular videos</em>",
		tokens: []TitleToken{
			{Type: TextToken, Text: "Popular videos", Tags: emphasisTag},
		},
	},
	{
		name:  "Placeholder Tag 1",
		title: "<em>{Game}</em> streamers recommended for you",
		tokens: []TitleToken{
			{Type: PlaceholderToken, Text: "Game", Tags: emphasisTag},
			{Type: TextToken, Text: " streamers recommended for you", Tags: noTags},
		},
	},
	{
		name:  "Placeholder Tag 2",
		title: "<link:directory>{Game}</link:directory>",
		tokens: []TitleToken{
			{Type: PlaceholderToken, Text: "Game", Tags: linkTag},
		},
	},
	{
		name:  "Multiple Tags 1",
		title: "Popular <link:directory><em>live channels</em></link:directory> in <em>{Game}</em>",
		tokens: []TitleToken{
			{Type: TextToken, Text: "Popular ", Tags: noTags},
			{Type: TextToken, Text: "live channels", Tags: linkAndEmphasisTag},
			{Type: TextToken, Text: " in ", Tags: noTags},
			{Type: PlaceholderToken, Text: "Game", Tags: emphasisTag},
		},
	},
	{
		name:  "Complex Unicode 1",
		title: "為您推薦的《<em>{Game}</em>》實況主",
		tokens: []TitleToken{
			{Type: TextToken, Text: "為您推薦的《", Tags: noTags},
			{Type: PlaceholderToken, Text: "Game", Tags: emphasisTag},
			{Type: TextToken, Text: "》實況主", Tags: noTags},
		},
	},
	{
		name:  "Complex Unicode 2",
		title: "추천 <em>{Game}</em> 스트리머",
		tokens: []TitleToken{
			{Type: TextToken, Text: "추천 ", Tags: noTags},
			{Type: PlaceholderToken, Text: "Game", Tags: emphasisTag},
			{Type: TextToken, Text: " 스트리머", Tags: noTags},
		},
	},
	{
		name:  "Crossed Tags 1",
		title: "<em>Hello</em><link:directory><em>there</em>!</link:directory>",
		tokens: []TitleToken{
			{Type: TextToken, Text: "Hello", Tags: emphasisTag},
			{Type: TextToken, Text: "there", Tags: linkAndEmphasisTag},
			{Type: TextToken, Text: "!", Tags: linkTag},
		},
	},
}

// TestTokenizerOnValidTitles tests that the tokenizer yields expected tokens given valid titles
func TestTokenizerOnValidTitles(t *testing.T) {
	for _, test := range tokenizerTests {
		t.Run(test.name, func(t *testing.T) {
			assert := assert.New(t)
			titleTokens, err := GetTitleTokens(test.title, "Default")
			assert.NoError(err, fmt.Sprintf("Did not properly tokenize test title %q", test.title))
			assert.Equal(test.tokens, titleTokens["Default"], "Expected title tokens did not match actual title tokens.")
		})
	}
}

var pluralTests = []struct {
	name    string
	title   string
	variant string
	tokens  map[string][]TitleToken
}{
	{
		name:    "Basic Test",
		title:   "{Number, plural, one {{Number} viewer} other {{Number} viewers}}",
		variant: "Default",
		tokens: map[string][]TitleToken{
			"Default::one": []TitleToken{
				{Type: PlaceholderToken, Text: "Number", Tags: noTags},
				{Type: TextToken, Text: " viewer", Tags: noTags},
			},
			"Default::other": []TitleToken{
				{Type: PlaceholderToken, Text: "Number", Tags: noTags},
				{Type: TextToken, Text: " viewers", Tags: noTags},
			},
		},
	},
	{
		name:    "Complex Unicode 1",
		title:   "{Number, plural, one {{Number} подаренная подписка} few {{Number} подаренные подписки} many {{Number} подаренных подписок} other {{Number} подаренной подписки}}",
		variant: "Default",
		tokens: map[string][]TitleToken{
			"Default::one": []TitleToken{
				{Type: PlaceholderToken, Text: "Number", Tags: noTags},
				{Type: TextToken, Text: " подаренная подписка", Tags: noTags},
			},
			"Default::few": []TitleToken{
				{Type: PlaceholderToken, Text: "Number", Tags: noTags},
				{Type: TextToken, Text: " подаренные подписки", Tags: noTags},
			},
			"Default::many": []TitleToken{
				{Type: PlaceholderToken, Text: "Number", Tags: noTags},
				{Type: TextToken, Text: " подаренных подписок", Tags: noTags},
			},
			"Default::other": []TitleToken{
				{Type: PlaceholderToken, Text: "Number", Tags: noTags},
				{Type: TextToken, Text: " подаренной подписки", Tags: noTags},
			},
		},
	},
	{
		name:    "Complex Unicode 2",
		title:   "{Number, plural, other {為您推薦的《<em>{Game}{Number}</em>》實況主}}",
		variant: "Default",
		tokens: map[string][]TitleToken{
			"Default::other": []TitleToken{
				{Type: TextToken, Text: "為您推薦的《", Tags: noTags},
				{Type: PlaceholderToken, Text: "Game", Tags: emphasisTag},
				{Type: PlaceholderToken, Text: "Number", Tags: emphasisTag},
				{Type: TextToken, Text: "》實況主", Tags: noTags},
			},
		},
	},
}

func TestTokenizerOnPluralTitles(t *testing.T) {
	for _, test := range pluralTests {
		t.Run(test.name, func(t *testing.T) {
			assert := assert.New(t)
			titleTokens, err := GetTitleTokens(test.title, test.variant)
			assert.NoError(err, fmt.Sprintf("Did not properly tokenize test title %q", test.title))
			assert.Equal(test.tokens, titleTokens, "Expected title tokens did not match actual title tokens.")
		})
	}
}

// TestTokenizerOnInvalidTitles tests that the tokenizer emits a failure if an invalid title is passed in
func TestTokenizerOnInvalidTitles(t *testing.T) {
	invalidTitleStrings := []string{
		"Popular <em>live channels<em>",
		"<e{Game}m>Popular videos</em>",
		"Popular <link:directory><123>live channels</em></link:directory> in <em>{Game}</em>",
		"<em>{Game{Tag}}</em> streamers recommended for you",
		"<link:directory>{Game}</em>",
		"<em>Videos</em> recommended for you{Game",
		"<em>Hello</em><link:directory><em>there</em>!<link:directory>",
		"추천 <em>{Game}</em> 스트리머 <end>",
	}
	assert := assert.New(t)
	for _, title := range invalidTitleStrings {
		_, err := GetTitleTokens(title, "Default")
		assert.Error(err, fmt.Sprintf("Did not catch expected error in title string %q.", title))
	}
}
