package titles

import (
	"code.justin.tv/amzn/TwitchTitles/tokenizer"
	"github.com/pkg/errors"
)

// buildResponseTokens converts the tokenizer tokens into the response title tokens
func buildResponseTokens(tokens []tokenizer.TitleToken, placeholderMap PlaceholderMap) ([]*TwitchTitleToken, error) {
	var responseTokens []*TwitchTitleToken
	var tokenError error
	var titleToken *TwitchTitleToken

	placeholderIndexTracker := make(map[PlaceholderType]int)
	for _, token := range tokens {
		switch token.Type {
		case tokenizer.TextToken:
			titleToken, tokenError = buildTitleTextToken(token)
		case tokenizer.PlaceholderToken:
			tokenType := getPlaceholderTokenType(token)
			titleToken, tokenError = buildTitlePlaceholderToken(placeholderMap, tokenType, placeholderIndexTracker[tokenType])
			placeholderIndexTracker[tokenType] += 1
		}
		if tokenError != nil {
			return nil, tokenError
		}
		responseTokens = append(responseTokens, titleToken)
	}
	return responseTokens, nil
}

// buildTitleTextToken uses the TagMap to build a title text token
func buildTitleTextToken(token tokenizer.TitleToken) (*TwitchTitleToken, error) {
	return &TwitchTitleToken{
		TokenType: TEXT_TOKEN_TYPE,
		TextToken: &TwitchTitleTokenText{
			Text:     token.Text,
			Location: getHyperlink(token.Tags),
			Emphasis: getEmphasis(token.Tags),
		},
	}, nil
}

// getHyperlink returns the link location for text token from its tags
func getHyperlink(tags []string) TwitchTitleTextTokenHyperlink {
	for _, tag := range tags {
		switch tag {
		case "link:directory":
			return CATEGORY_DIRECTORY_LINK
		case "link:streams":
			return STREAMS_DIRECTORY_LINK
		}
	}
	return NO_LINK
}

// getEmphasis returns if a text token should be emphasized from its tags
func getEmphasis(tags []string) bool {
	for _, tag := range tags {
		if tag == "em" {
			return true
		}
	}
	return false
}

// getPlaceholderTokenType maps a placeholder token's text to a PlaceholderType
func getPlaceholderTokenType(token tokenizer.TitleToken) PlaceholderType {
	switch token.Text {
	case "Category":
		return CategoryPlaceholder
	case "Streamer":
		return StreamPlaceholder
	case "Tag":
		return TagPlaceholder
	case "Number":
		return NumberPlaceholder
	case "Date":
		return DatePlaceholder
	default:
		return UnknownPlaceholder
	}
}

// buildTitlePlaceholderToken builds a placeholder token from the placeholderMap
func buildTitlePlaceholderToken(placeholderMap PlaceholderMap, placeholderType PlaceholderType, placeholderIndex int) (*TwitchTitleToken, error) {
	placeholders, ok := placeholderMap[placeholderType]
	if !ok {
		return nil, errors.Errorf("could not find placeholder for type = %s", placeholderType)
	}

	if len(placeholders) <= placeholderIndex {
		return nil, errors.Errorf("not enough placeholders of type = %s", placeholderType)
	}

	var tokenType TwitchTitleTokenType
	var tokenErr error
	switch placeholderType {
	case CategoryPlaceholder:
		tokenType = CATEGORY_TOKEN_TYPE
	case StreamPlaceholder:
		tokenType = STREAM_TOKEN_TYPE
	case TagPlaceholder:
		tokenType = TAG_TOKEN_TYPE
	case NumberPlaceholder:
		tokenType = NUMBER_TOKEN_TYPE
	case DatePlaceholder:
		tokenType = DATE_TOKEN_TYPE
	default:
		tokenErr = errors.Errorf("could not insert placeholder for type = %s", placeholderType)
	}

	return &TwitchTitleToken{
		TokenType:        tokenType,
		PlaceholderToken: &placeholders[placeholderIndex],
	}, tokenErr
}
