package lib

import (
	"context"
	"os"
	"time"

	"github.com/twitchtv/twirp"
)

// ServerHooks provides a new Twirp ServerHook to use in Twirp server
func (a *AccessLog) ServerHooks() *twirp.ServerHooks {
	hostname := ""
	if osHostname, err := os.Hostname(); err == nil {
		hostname = osHostname
	}

	return &twirp.ServerHooks{
		Error: func(ctx context.Context, err twirp.Error) context.Context {
			ctx = context.WithValue(ctx, accessLogErrKey, err.Error())
			return ctx
		},
		RequestReceived: func(ctx context.Context) (context.Context, error) {
			ctx = context.WithValue(ctx, requestTimeKey, time.Now())
			ctx = WithTrace(ctx)
			return ctx, nil
		},
		ResponseSent: func(ctx context.Context) {
			msg := &Message{
				MsgTime:  time.Now(),
				Hostname: hostname,
				TraceID:  twirpTraceID(ctx),
			}

			if method, ok := twirp.MethodName(ctx); ok {
				msg.Method = method
			}

			if code, ok := twirp.StatusCode(ctx); ok {
				msg.StatusCode = code
			}

			if recvTime, ok := ctx.Value(requestTimeKey).(time.Time); ok {
				duration := time.Since(recvTime)
				msg.ReceivedTime = recvTime
				msg.DurationMS = duration.Nanoseconds() / time.Millisecond.Nanoseconds()
			}

			if errMsg, ok := ctx.Value(accessLogErrKey).(string); ok {
				msg.Error = errMsg
			}

			a.Log(ctx, msg)
		},
	}
}

// twirpTraceID returns the X-Ray trace header if one exists.
func twirpTraceID(ctx context.Context) string {
	headers, exists := twirp.HTTPRequestHeaders(ctx)
	if exists {
		return headers.Get("X-Amz-Trace-Id")
	}
	return ""
}
