package kswriter

import (
	"context"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

// newErrHandler returns an errHandler struct which must be started manually via the run method.
func newErrHandler(kc minimalKinesis, cfg Settings, logger logger, stats Statter) *errHandler {
	return &errHandler{
		ct:                0,
		decr:              make(chan bool, 1),
		errBuff:           make(chan kinesis.PutRecordsInput, cfg.ErrorBufferSize),
		errMsg:            make(chan kinesis.PutRecordsInput, 20),
		procChan:          make(chan bool, 1),
		kc:                kc,
		errorBufferSize:   cfg.ErrorBufferSize,
		kinesisStreamName: cfg.KinesisStreamName,
		logger:            logger,
		stats:             stats,
	}
}

// errHandler holds a buffer for kinesis.PutRecordsInput and
// manages the buffer such that oldest records are discarded
// once the max buffer size is reached and attempts to process
// the records in the queue as defined in errHandler.run
//
// Add to the buffer by calling errorBuffer.newError(error, kinesis.PutRecordsInput)
//
// errHandler components:
// - ct: counter that holds how many records are in the buffer
// - decr: channel that is listed to decrement the ct counter
// - errBuff: buffered channel that holds the records
// - errMsg: channel that temporarily houses incoming records
// - procChan: a channel that is used to determine if errHandler.procBuff has
//   completed
// - kc: a *kinesis.Kinesis that is used by errHandler.procBuff for making
//   Amazon kinesis put requests (keep the garbage generation to a minimum)
//
type errHandler struct {
	ct                int64
	decr              chan bool
	errBuff           chan kinesis.PutRecordsInput
	errMsg            chan kinesis.PutRecordsInput
	procChan          chan bool
	kc                minimalKinesis
	errorBufferSize   int64
	kinesisStreamName string
	logger            logger
	stats             Statter
}

// AddError is called to add a new record to the error buffer
// to do: inspect the error message to determine an appropriate action
func (e *errHandler) AddError(err error, kr kinesis.PutRecordsInput) {
	e.errMsg <- kr
}

// run listens for new error messages, stores them in a buffered channel, and
// then periodically processes any errors in the buffered channel
func (e *errHandler) run(ctx context.Context) {
	defer close(e.errBuff)
	defer close(e.errMsg)
	defer close(e.decr)
	defer close(e.procChan)
	// check to see if errors are loaded up
	tick := time.NewTimer(time.Second * 5)
	defer tick.Stop()
	for {
		select {
		case <-ctx.Done():
			return
		case <-tick.C:
			tick.Stop()
			// has to be non blocking as it may run for multiple seconds
			go e.procBuff(ctx)
		case <-e.procChan:
			tick.Reset(time.Second * 5)
		case <-e.decr:
			e.ct = e.ct - 1
		case ev := <-e.errMsg:
			e.stats.Count("streamlog.errorbuff.new", 1)
			e.addEvent(ev)
		}
	}
}

// addEvent is responsible for making sure the buffer is full
// and if errHandler.ct == ErrorBufferSize then remove an event
// from the buffer and then add the new one
func (e *errHandler) addEvent(kr kinesis.PutRecordsInput) {
	switch {
	case e.ct == e.errorBufferSize:
		<-e.errBuff
		e.errBuff <- kr
	default:
		e.ct++
		e.errBuff <- kr
	}
}

// errHandler.procBuff processes the messages for a fixed time interval
// by taking the kinesis Records from the errBuff and then trying to
// resend them to kinesis
func (e *errHandler) procBuff(ctx context.Context) {
	if !e.chkKinesisStat(ctx, e.kc) {
		return
	}
	loop := true
	for loop {
		select {
		case <-ctx.Done():
			return
		case kr := <-e.errBuff:
			e.stats.Count("streamlog.kinesis.put", 1)
			resp, err := e.kc.PutRecordsWithContext(ctx, &kr)
			switch {
			case err != nil:
				e.stats.Count("streamlog.kinesis.fail", 1)
				// record is going to get tossed
				e.logger.Error(
					"streamlog.errHandler.procBuff",
					"got err from kinesis: %s",
					err.Error(),
				)
				// break out of loop
				loop = false
			default:
				e.logger.Info(
					"streamlog.errHandler.procBuff",
					"failed record count: %d",
					*resp.FailedRecordCount,
				)
				e.decr <- true
			}
		default:
			// if no records in errBuff break out of loop
			loop = false
		}
	}
	// turn ticker back on
	e.procChan <- true
	e.stats.Count("streamlog.errorbuff.procbuffct", 1)
}

// chkKinesisStat makes a req and then waits for either a timeout
// or a return from the request
func (e *errHandler) chkKinesisStat(ctx context.Context, kc minimalKinesis) bool {
	tmr := time.NewTimer(time.Second)
	c := make(chan bool, 1)
	go e.chkKinesisStatReq(ctx, kc, c)
	select {
	case b := <-c:
		return b
	case <-tmr.C:
		tmr.Stop()
		return false
	}
}

// chkKinsisStatReq makes a Describe Stream API call to the kinesis API
// and if the StreamDescription is nil or there is an error in the returned
// req it returns false
func (e *errHandler) chkKinesisStatReq(ctx context.Context, kc minimalKinesis, c chan bool) {
	out, err := kc.DescribeStreamWithContext(
		ctx,
		&kinesis.DescribeStreamInput{
			Limit:      aws.Int64(1),
			StreamName: &e.kinesisStreamName,
		},
	)
	switch {
	case err != nil:
		c <- false
	case out.StreamDescription == nil:
		c <- false
	default:
		c <- true
	}
}
