package kswriter

import (
	"context"
	"errors"
	"runtime"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

type testKinesis struct {
}

func (t *testKinesis) PutRecordsWithContext(ctx aws.Context, input *kinesis.PutRecordsInput, opts ...request.Option) (*kinesis.PutRecordsOutput, error) {
	return putRecords(), nil
}

func (t *testKinesis) DescribeStreamWithContext(ctx aws.Context, input *kinesis.DescribeStreamInput, opts ...request.Option) (*kinesis.DescribeStreamOutput, error) {
	return describeStream(), nil
}

// TestErrHandler makes sure that the errHandler takes messages
// and ensures that the latest records are added while oldest
// records are discarded if the amount pushed into the buffer
// exceeds the max size of the error buffer
func TestErrHandler(t *testing.T) {
	runtime.GOMAXPROCS(2)
	cfg := Settings{
		ErrorBufferSize:      1000,
		DrainEventQueueTimer: 10 * time.Second,
		KinesisStreamName:    "KinesisStreamName",
	}
	ctx, stop := context.WithCancel(context.Background())
	t.Cleanup(stop)
	e := newErrHandler(&testKinesis{}, cfg, &nopLogger{}, &NopStatter{})
	go e.run(ctx)
	kr := getNewPutRecordsInput(cfg.KinesisStreamName)
	for i := 0; i < int(cfg.ErrorBufferSize*2); i++ {
		e.AddError(errors.New("new error"), kr)
	}
	ct := 0
	tick := time.NewTicker(time.Second)
	t.Cleanup(tick.Stop)
	chk := true
	for chk {
		select {
		case <-e.errBuff:
			ct++
		case <-tick.C:
			if ct != int(cfg.ErrorBufferSize) {
				t.Log("TestErrHandler: event ct should equal errorBufferSize")
				t.Fail()
			}
			chk = false
		default:
		}
	}
}

// TestErrHandlerToKinesis tests to ensure that errors in the errorBuffer do
// get sent to mockKinesis.  Loads up the error buffer with error messages and then
// waits to give it time to drain the error buffer and makes sure that there
// are not any records left in the error buffer.
func TestErrHandlerToKinesis(t *testing.T) {
	runtime.GOMAXPROCS(2)
	cfg := Settings{
		ErrorBufferSize:      1000,
		DrainEventQueueTimer: 10 * time.Second,
		KinesisStreamName:    "KinesisStreamName",
	}
	ctx, stop := context.WithCancel(context.Background())
	t.Cleanup(stop)
	e := newErrHandler(&testKinesis{}, cfg, &nopLogger{}, &NopStatter{})
	go e.run(ctx)
	kr := getNewPutRecordsInput(cfg.KinesisStreamName)
	for i := 0; i < int(cfg.ErrorBufferSize*2); i++ {
		e.AddError(errors.New("new error"), kr)
	}
	time.Sleep(time.Second * 10)
	tick := time.NewTicker(time.Second)
	chk := true
	for chk {
		select {
		case <-e.errBuff:
			chk = false
			t.Log("TestErrHandlerToKinesis: had events in the error buffer")
			t.Fail()
		case <-tick.C:
			chk = false
		}
	}
}
